#define __DACT_C
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "dact.h"
#include "parse.h"
#include "dendian.h"
#include "math.h"
#include "algorithms.h"
#include "module.h"
#ifdef CHECK_VERSION
#include "net.h"
#endif

extern char *optarg;
extern int optind, opterr, optopt;
uint32_t DACT_BLK_SIZE=4088;

int print_help(int argc, char **argv) {
	printf("DACT %i.%i.%i-%s by Netfuel Design. <dact@rkeene.org>\n", DACT_VER_MAJOR, \
		DACT_VER_MINOR, DACT_VER_REVISION,
#ifdef DEBUG
		"dev"
#else
		"rel"
#endif
		);
	printf("usage: %s [-dsfcvlCh] [-b NN] [-e NN] [-m CONF] [file ...]\n",argv[0]);
	printf("  -d          Decompress instead of compressing.\n");
	printf("  -s          Give statistics rather than compress or decompress.\n");
	printf("  -f          Force unsafe things to happen.\n");
	printf("  -c          (De)compress to stdout.\n");
	printf("  -v          Increase verbosity.\n");
	printf("  -l          List available algorithms.\n");
/*	printf("  -C          Complain when compression errors occur.\n");
*/
	printf("  -h          Give this help.\n");
	printf("  -b NN       Use a block size of NN bytes for compression.\n");
	printf("  -e NN       Exclude algorithm NN from being used.\n");
	printf("  -m CONF     Load config file CONF.\n");
	printf("  file...     File(s) to (de)compress.  (If none given, use standard input).\n");
	return(0);
}

void dact_loadconfig(const char *path, char *options) {
	char *line=NULL, *item_buf[4]={NULL, NULL, NULL, NULL};
	int i;
	FILE *cfd;

	if ((cfd=fopen(path,"r"))==NULL) return;
	while (!feof(cfd)) {
		line=malloc(512);
		fgets(line, 512, cfd);
		i=(strlen(line)-1);
		if (line[i]=='\n') line[i]=0;
		for (i=0;i<4;i++) item_buf[i]=NULL;
		for (i=0;i<4;i++) {
			if ((item_buf[i]=strsep(&line, "\t "))==NULL)  break;
			if (item_buf[i][0]==0) i--;
		}
		if (item_buf[0]==NULL) continue;

		if (!strcmp(item_buf[0],"version_check")) {
			options[DACT_OPT_VERCHK]=!!strcmp(item_buf[1],"off");
		}
		if (!strcmp(item_buf[0],"binary_check")) {
			options[DACT_OPT_BINCHK]=!!strcmp(item_buf[1],"off");
		}
		if (!strcmp(item_buf[0],"module_dir")) {
			strncpy(moduledirectory,item_buf[1],127);
		}
		if (!strcmp(item_buf[0],"load_module")) {
			init_modules();
			load_module(item_buf[1]);
		}



		free(line);
	}
	fclose(cfd);
}



uint32_t is_latest (const char *options) {
#if defined(CHECK_VERSION)
	int fd;
	char ver_maj[4]={0,0,0,0}, ver_min[4]={0,0,0,0}, ver_rev[4]={0,0,0,0};
	char bigbuf[1024];
	int vers[3];

	if (options[DACT_OPT_VERCHK]==0) return(0);
	if (getuid()==0) return(0);

	if ((fd=createconnection_tcp("www.rkeene.org", 80))<0) return(0);

	write(fd, "GET http://www.rkeene.org/devel/dact/VERSION\n", 45);
	read(fd, &bigbuf,1024);

	memcpy(ver_maj,bigbuf,3);
	memcpy(ver_min,bigbuf+3,3);
	memcpy(ver_rev,bigbuf+6,3);

	closeconnection(fd);

	vers[0]=atoi(ver_maj);
	vers[1]=atoi(ver_min);
	vers[2]=atoi(ver_rev);

	if ( ((vers[0]<<16)|(vers[1]<<8)|vers[2]) > ((DACT_VER_MAJOR<<16)|(DACT_VER_MINOR<<8)|DACT_VER_REVISION) ) {
		return((vers[0]<<16)|(vers[1]<<8)|vers[2]);
	} else {
		return(0);
	}
#else
	return(0);
#endif
}


int dact_shutdown(int retval) {
	unload_modules();
	return(retval);
}

char *dact_getoutfilename(const char *orig, const int mode) {
	char *ret=NULL;

	switch (mode) {
		case DACT_MODE_COMPR:
			ret=malloc(strlen(orig)+5);
			strcpy(ret,orig);
			strcat(ret,".dct");
			break;
		case DACT_MODE_DECMP:
			if (strcmp(&orig[strlen(orig)-4],".dct")) {
				return(NULL);
			}
			ret=malloc(strlen(orig)-3);
			strncpy(ret,orig,strlen(orig)-4);
			break;
		case DACT_MODE_STAT:
			return(NULL);
			break;
	}
	return(ret);
}

uint32_t dact_blk_decompress(char *ret, const char *srcbuf, const uint32_t size, const char options, const int algo) {
	uint32_t retval;

	if (algo==0xff) return(0);

	if (algorithms[algo]==NULL) {
		PRINTERR("Algorithm unavailble.");
		return(0);
	}

	retval=algorithms[algo](DACT_MODE_DECMP, NULL, srcbuf, ret, size);

	return(retval);
}


uint32_t dact_blk_compress(char *algo, char *ret, const char *srcbuf, const uint32_t size, const char *options) {
	char *tmpbuf, *smallbuf=NULL;
	int i, highest_algo=0;
	char smallest_algo;
	uint32_t smallest_size=-1, x;

	if ((tmpbuf=malloc(size*2))==NULL) { PERROR("malloc"); return(0); }

	for (i=0;i<256;i++) {
		if (algorithms[i]!=NULL && algorithms[i]!=DACT_FAILED_ALGO) highest_algo=i;
	}

	for (i=0;i<=highest_algo;i++) {
		if (algorithms[i]!=NULL && algorithms[i]!=DACT_FAILED_ALGO) {
			x=algorithms[i](DACT_MODE_COMPR, NULL, srcbuf, tmpbuf, size);
			if ((x<smallest_size || smallest_size==-1) && x!=-1) {
				smallest_size=x;
				smallest_algo=i;
				if (smallbuf!=NULL) free(smallbuf);
				if ((smallbuf=malloc(smallest_size))==NULL) { PERROR("malloc"); free(tmpbuf); return(0); }
				memcpy(smallbuf, tmpbuf, smallest_size);
			}

			if (options[DACT_OPT_VERB]>1) {
				PRINT_LINE; fprintf(stderr, "dact: \033[%im----| %03i  | %-7i | %s\033[0m\n", (smallest_algo==i)*7 , i, x, algorithm_names[i]);
			}

		}
	}

	free(tmpbuf);
	memcpy(algo, &smallest_algo, sizeof(char));
	memcpy(ret, smallbuf, smallest_size);
	return(smallest_size);
}

uint32_t dact_process_file(const int src, const int dest, const int mode, const char *options, const char *filename) {
	unsigned char algo;
	char *in_buf, *out_buf;
	char version[3]={DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION};
	char file_opts=0;
	uint32_t bytes_read, retsize;
	uint32_t filesize=0, blk_cnt=0, file_extd_size=0, blksize=0, fileoutsize=0;
	uint32_t magic=0;
	int blksize_size;
	int x=0;

	if (mode==DACT_MODE_COMPR) {
		if (((in_buf=malloc(DACT_BLK_SIZE))==NULL) || \
			((out_buf=malloc(DACT_BLK_SIZE*2))==NULL)) {
				PERROR("malloc");
				return(-1);
		}

		blksize_size=BYTESIZE(DACT_BLK_SIZE);

		write_de(dest, DACT_MAGIC_NUMBER, 4);
		write(dest, &version[0], 1);
		write(dest, &version[1], 1);
		write(dest, &version[2], 1);
		write_de(dest, 0, 4); /* Place holder for ORIG FILE SIZE */
		write_de(dest, 0, 4); /* Place holder for NUM BLOCKS */
		write_de(dest, DACT_BLK_SIZE, 4);
		write_de(dest, file_opts, 1); /* XXX: Option byte... Or not? */
		write_de(dest, file_extd_size, 4); /* Place holder for SIZEOF EXTENDED DTA */


		if (options[DACT_OPT_VERB]) {
			PRINTERR("Blk | Algo | Size    | Name");
			PRINTERR("----+------+---------+---------------------------");
		}

		memset(in_buf,0,DACT_BLK_SIZE);
		while ( (bytes_read=read(src, in_buf, DACT_BLK_SIZE))>0) {
			filesize+=bytes_read;
			blk_cnt++;
			retsize=dact_blk_compress(&algo, out_buf, in_buf, DACT_BLK_SIZE, options);
			if (retsize>0) {
				if (options[DACT_OPT_VERB]) {
					if (options[DACT_OPT_VERB]>1) {
						PRINTERR("^^^\\ /^^^^\\ /^^^^^^^\\ /^^^^^^^^^^^^^^^^^^^^^^^^^^");
					}
					PRINT_LINE; fprintf(stderr, "dact: %03i | %03i  | %-7i | %s\n",blk_cnt,algo,retsize,algorithm_names[algo]);
					if (options[DACT_OPT_VERB]>1) {
						PRINTERR("___/ \\____/ \\_______/ \\__________________________");
					}
				}

				write(dest, &algo, 1);
				write_de(dest, retsize, blksize_size);

				if (write(dest, out_buf, retsize)!=retsize) {
					PERROR("write");
					free(in_buf);
					free(out_buf);
					return(0);
				}
			} else {
				PRINTERR("Compression resulted in 0-byte block.");
				free(in_buf);
				free(out_buf);
				return(0);
			}
			memset(in_buf,0,DACT_BLK_SIZE);
		}

		if (bytes_read<0) {
			PERROR("read");
		}

		free(in_buf);
		free(out_buf);

		if (lseek(dest, 7, SEEK_SET)<0) {
/* If we can't rewind the stream, put magic+fileisze */
			write_de(dest, DACT_MAGIC_PEOF, 4);
			write_de(dest, filesize, 4);
		} else {
			write_de(dest, filesize, 4);
			write_de(dest, blk_cnt, 4);
		} 
		

		return(filesize);
	}

	if (mode==DACT_MODE_DECMP) {

		read_de(src, &magic, 4);

		if (magic!=DACT_MAGIC_NUMBER) {
			PRINTERR("Bad magic.");
			return(-1);
		}

		read(src, &version[0], 1);
		read(src, &version[1], 1);
		read(src, &version[2], 1);
		read_de(src, &filesize, 4);
		read_de(src, &blk_cnt, 4);
		read_de(src, &DACT_BLK_SIZE, 4);
		read(src, &file_opts, 1);
		read_de(src, &file_extd_size, 4);
/* For now, we skip over extended data. */
		lseek(src, file_extd_size, SEEK_CUR);

/* 
   XXX: Even if we don't resolve it here, we can resolve it later...
   Should we even bother to do it here if we can?

   XXX: When CAN'T we rewind a read stream?

*/
		if (filesize==0) {
/* See if we can rewind our stream, so when we get to the end, we can come back! */
			if (lseek(src, 0, SEEK_SET)==0) {
				lseek(src, -8, SEEK_END);
				read_de(src, &magic, 4);
				if (magic!=DACT_MAGIC_PEOF) {
					PRINTERR("File is corrupt.");
					return(-1);
				}
				read_de(src, &filesize, 4);
				lseek(src, 24+file_extd_size, SEEK_SET);
			}
		}


		if (((out_buf=malloc(DACT_BLK_SIZE))==NULL) ) {
				PERROR("malloc");
				return(-1);
		}


		blksize_size=BYTESIZE(DACT_BLK_SIZE);

		while (1) {
			if (read(src, &algo, 1)==0) break;
			if (algo==0xff) break; /* 0xff is reserved for EOF */

			read_de(src, &blksize, blksize_size);

			if ((in_buf=malloc(blksize))==NULL) {
				PERROR("malloc");
				free(out_buf);
				return(-1);
			}

			read(src, in_buf, blksize);

			if (filesize==0) {
				read_de(src, &magic, 4);
				read_de(src, &filesize, 4);
				if (read(src, &x, 1)==0) {
					if (magic!=DACT_MAGIC_PEOF) {
						PRINTERR("Corrupt stream.");
						free(in_buf);
						free(out_buf);
						return(-1);
					}
				} else {
					lseek(src, -9, SEEK_CUR);
					filesize=0;
				}
			}



			if ((bytes_read=dact_blk_decompress(out_buf, in_buf, blksize, 0, algo))==0) {
				PRINTERR("Decompression resulted in 0-byte block.");
			}
			fileoutsize+=bytes_read;


			if (fileoutsize>filesize && filesize!=0) {
				write(dest, out_buf, DACT_BLK_SIZE-(fileoutsize-filesize));
			} else {
				write(dest, out_buf, bytes_read);
			}



			free(in_buf);

			
		}

		free(out_buf);


		return(filesize);

	}


	if (mode==DACT_MODE_STAT) {
		read_de(src, &magic, 4);
		read(src, &version[0], 1);
		read(src, &version[1], 1);
		read(src, &version[2], 1);
		read_de(src, &filesize, 4);
		read_de(src, &blk_cnt, 4);
		read_de(src, &DACT_BLK_SIZE, 4);
		read(src, &file_opts, 1);
		read_de(src, &file_extd_size, 4);

		printf("File              :   %s\n", filename);
		printf("Magic             :   0x%08x",magic);
		if (magic!=DACT_MAGIC_NUMBER) {
			printf(" (bad magic)\n");
			return(0);
		} else {
			printf("\n");
		}

		if (filesize==0) {
			lseek(src, -8, SEEK_END);
			read_de(src, &magic, 4);
			read_de(src, &filesize, 4);
			if (magic!=DACT_MAGIC_PEOF) {
				PRINTERR("Bad magic, corrupt stream.");
				return(0);
			}
		}
		fileoutsize=lseek(src, 0, SEEK_END);

		printf("Dact Version      :   %i.%i.%i\n",version[0],version[1],version[2]);
		printf("Block Size        :   %i\n", DACT_BLK_SIZE);
		printf("Block Header Size :   %i\n", BYTESIZE(DACT_BLK_SIZE)+1);
		printf("Compressed Size   :   %i\n", fileoutsize);
		printf("Uncompressed Size :   %i\n", filesize);
		printf("Ratio             :   %2.3f to 1.0\n", ((float) filesize)/((float) fileoutsize) );
		printf("\n");


		return(1);
	}

	return(0);
}

int main(int argc, char **argv) {
	unsigned char options[8]={0,0,0,0,0,0,0,0};
	signed char opt;
	char **in_files, *in_file=NULL, *out_file=NULL;
	char dact_binfile[256];
	int filecnt=0;
	int in_fd, out_fd;
	int mode=DACT_MODE_COMPR;
	uint32_t i;


	dact_loadconfig(DACT_CONFIG_FILE, options);
#ifndef GO32
	strncpy(dact_binfile,getenv("HOME"),sizeof(dact_binfile)-1);
	strncat(dact_binfile,"/.dact/dact.conf",sizeof(dact_binfile)-strlen(dact_binfile)-1);
	dact_loadconfig(dact_binfile, options);
#endif

	if (options[DACT_OPT_BINCHK]) {
#ifdef GO32
		strcpy(dact_binfile,"c:/dact/dact.exe");
#else
		strncpy(dact_binfile,getenv("HOME"),sizeof(dact_binfile)-1);
		strncat(dact_binfile,"/.dact/dact.bin",sizeof(dact_binfile)-strlen(dact_binfile)-1);
#endif

		if (strcmp(argv[0],dact_binfile)) {
			if (!access(dact_binfile,X_OK)) {
				argv[0]=dact_binfile;
				execv(dact_binfile,argv);
			}
		}
	}
 
	while ((opt=getopt(argc,argv,"dfsvcCm:e:lb:h"))!=-1) {
		switch (opt) {
			case 'd':
				mode=DACT_MODE_DECMP;
				break;
			case 'f':
				options[DACT_OPT_FORCE]++;
				break;
			case 's':
				mode=DACT_MODE_STAT;
				break;
			case 'c':
				options[DACT_OPT_STDOUT]=!options[DACT_OPT_STDOUT];
				break;
			case 'b':
				i=atoi2(optarg);
				if (i<DACT_BLK_SIZE_MAX) DACT_BLK_SIZE=i;
				break;
			case 'v':
				options[DACT_OPT_VERB]++;
				break;
			case 'C':
				options[DACT_OPT_COMPLN]++;
				break;
			case 'm':
				dact_loadconfig(optarg, options);
				break;
			case 'e':
				i=(atoi(optarg)&0xff);
				algorithms[i]=DACT_FAILED_ALGO;
				break;
			case 'l':
				for (i=0;i<255;i++) {
					if (algorithms[i]==NULL || algorithms[i]==DACT_FAILED_ALGO) continue;
					fprintf(stderr, "%3i - %s\n",i,algorithm_names[i]);
			
				}
				return(0);
				break;
			case '?':
			case 'h':
				return(print_help(argc,argv));
		}

	}

/* Check for a new version of DACT (is_latest() returns immediately if
 * this option is disabled.)
 */
	if ((i=is_latest(options))) {
		PRINTERR("**+");
		PRINTERR("**> There is a new version of DACT available.");
		PRINTERR("**>");
		PRINT_LINE; fprintf(stderr, "dact: **> [CURR: DACT %i.%i.%i]\n",DACT_VER_MAJOR, DACT_VER_MINOR, DACT_VER_REVISION);
		PRINT_LINE; fprintf(stderr, "dact: **> [NEW:  DACT %i.%i.%i]\n",i>>16,(i>>8)&0xff,i&0xff);
		PRINTERR("**>");
		PRINTERR("**> Run `dact-upgrade.sh' if you installed it to get it.");
		PRINTERR("**> or get the source at: http://www.rkeene.org/devel/dact.tar.gz");
		PRINTERR("**>");
		PRINTERR("**-");
	}


	in_files=&argv[optind];

/* Loop through extra parameters (files ...) and setup FDs for them */
	do {
		in_fd=-1;
		out_fd=-1;

		in_file=in_files[filecnt];
		if (in_file!=NULL) {
/* Determine resulting file name */
			out_file=dact_getoutfilename(in_file,mode);
			if ((in_fd=open(in_file,O_RDONLY))==-1) {
				fprintf(stderr, "dact: Can't open %s.\n",in_file);
				PERROR("open");
				continue;
			}
			if (out_file!=NULL) {
				if (access(out_file,F_OK)!=-1 && options[DACT_OPT_FORCE]==0 && options[DACT_OPT_STDOUT]==0) {
					fprintf(stderr, "dact: %s exists.\n",out_file);
					close(in_fd);
					continue;
				}
				if (options[DACT_OPT_STDOUT]) {
					out_fd=STDOUT_FILENO;
				} else {
					if ((out_fd=open(out_file,O_WRONLY|O_CREAT|O_TRUNC,0644))==-1) {
						fprintf(stderr, "dact: Can't open %s for writing.\n",out_file);
						PERROR("open");
						continue;
					}
				}
			}
		}

/* Use STDIN/STDOUT if no files specified ... */
		if (in_file==NULL && filecnt==0) {
/* ... But only if STDOUT isn't a terminal */
			if (isatty(STDOUT_FILENO) && options[DACT_OPT_FORCE]==0) {
				fprintf(stderr, "dact: Refusing to write compressed output to a terminal.\n");
			} else {
				out_fd=STDOUT_FILENO;
				in_fd=STDIN_FILENO;
			}
		}

/* Okay, we're all done, now pass these to something to do the real stuff */
		if (in_fd!=-1 && (out_fd!=-1 || mode==DACT_MODE_STAT)) {
			if (dact_process_file(in_fd, out_fd, mode, options, in_file)==0) {
				close(in_fd);
				close(out_fd);
				return(dact_shutdown(-1));
			}
		}
/* Cleanup */
	
		if (out_fd!=-1) close(out_fd);
		if (in_fd!=-1) close(in_fd);

	} while (in_files[filecnt++]!=NULL);

	return(dact_shutdown(0));
}
