#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#define GP2P_VERSION PACKAGE_VERSION

#define GAIM_PLUGINS
#include <gaim.h>

#include <version.h>
#include <signals.h>

#include <gtk/gtk.h>
#include <gtkgaim.h>
#include <util.h>
#include <gtkplugin.h>
#include <gtkutils.h>

#include <string.h>
#include <ctype.h>

#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif

#define GP2P_PLUGIN_ID "gp2p"
#define GP2P_WEBSITE "http://www.rkeene.org/oss/gp2p/"

#define GP2P_TYPE_CMD 275268
#define GP2P_TYPE_RET 340308

#define GP2P_CMD_SEARCH 0xc50530c8

static int gp2p_loaded = 0;

static uint32_t valhash(const char *val) {
	uint32_t ret = 0;
	const char *tmp = NULL;

	for (tmp = val; *tmp != '\0'; tmp++) {
		ret <<= 6;
		ret |= toupper(*tmp);
	}

	return(ret);
}

static int parmcpy(char *dest, const char *src, const char *parm, const char *highptr, size_t destlen) {
	size_t tmplen = 0;
	char *tmp = NULL;

	/* Locate the CMD parameter. */
	tmp = strstr(src, parm);
	if (tmp == NULL || tmp > highptr) {
		return(-1);
	}
	tmp = tmp + strlen(parm); /* Skip past the `PARM=' part. */

	/* Ignore quotes. */
	if (*tmp == '\"') {
		tmp++;
	} 

	tmplen = strcspn(tmp, "\t\n\" ]>");
	if (tmplen >= destlen) {
		tmplen = destlen - 1;
	}
	strncpy(dest, tmp, tmplen);

	dest[destlen - 1] = '\0';

	return(0);
}


/*

   <CMD ttl=INT host=HOST:PORT cmd=COMMAND>...</CMD>

 */
static char *gp2p_handle_cmd(char *data, const char *msgid) {
	uint32_t cmdid = 0;
	uint16_t ttl = 0;
	char cmd[128] = {0}, host[128] = {0}, ttlstr[32] = {0};
	char *lowptr = NULL, *highptr = NULL, *message = NULL;
	int pc_ret = -1;

	lowptr = strstr(data, "[CMD ");
	if (lowptr == NULL) {
		return(NULL);
	}

	highptr = strstr(lowptr, "[/CMD]");
	if (highptr == NULL) {
		return(NULL);
	}
	*highptr = '\0';

	highptr = strchr(lowptr, ']');
	if (highptr == NULL) {
		return(NULL);
	}

	/* CMD field. */
	pc_ret = parmcpy(cmd, data, "cmd=", highptr, sizeof(cmd));
	if (pc_ret < 0) {
		return(NULL);
	}

	/* TTL field. */
	pc_ret = parmcpy(ttlstr, data, "ttl=", highptr, sizeof(ttlstr));
	if (pc_ret < 0) {
		return(NULL);
	}
	ttl = atoi(ttlstr);
	if (ttl == 0 || ttl > 20) {
		return(NULL);
	}

	/* HOST field. */
	pc_ret = parmcpy(host, data, "host=", highptr, sizeof(host));
	if (pc_ret < 0) {
		return(NULL);
	}

	message = highptr;
	if (message == NULL) {
		return(NULL);
	}
	message++;

	cmdid = valhash(cmd);

	switch (cmdid) {
		case GP2P_CMD_SEARCH:
			printf("P2P: Executing search: [ttl=%i, host=%s, id=%s] with parms: %s\n", ttl, host, msgid, message);
			break;
		default:
			printf("P2P: Executing command: %s (0x%x) [ttl=%i, host=%s] with parms: %s\n", cmd, cmdid, ttl, host, message);
			break;
	}

	return(NULL);
}

/*

   <P2P type="cmd" id="abcdef">...</P2P>

 */
static int gp2p_handle_mesg(const char *source, char *message) {
	uint32_t typeid = 0;
	char *messagedata = NULL, *highptr = NULL;
	char type[128] = {0}, msgid[128] = {0};
	char *tmp = NULL;
	char *cmdret = NULL;
	int pc_ret = -1;

//	printf("Check Message from %s: %s\n", source, message);

	/* Ensure that we have some valid data. */
	if (message == NULL || source == NULL) {
		return(0);
	}

	/* We only handle our type of messages. */
	if (strncmp("[P2P ", message, 5) != 0) {
		return(0);
	}
	tmp = strstr(message, "[/P2P]");
	if (tmp == NULL) {
		return(0);
	}
	*tmp = '\0';

	highptr = strchr(message, ']');
	if (highptr == NULL) {
		return(1);
	}

	/* TYPE field. */
	pc_ret = parmcpy(type, message, "type=", highptr, sizeof(type));
	if (pc_ret < 0) {
		return(1);
	}

	/* ID field. */
	pc_ret = parmcpy(msgid, message, "id=", highptr, sizeof(msgid));
	if (pc_ret < 0) {
		return(1);
	}

	messagedata = highptr;
	if (messagedata == NULL) {
		return(1);
	}

	messagedata++;

	typeid = valhash(type);
	switch (typeid) {
		case GP2P_TYPE_CMD:
			cmdret = gp2p_handle_cmd(messagedata, msgid);
			break;
		case GP2P_TYPE_RET:
		default:
			printf("P2P Message from %s (type=%s, 0x%x): %s\n", source, type, typeid, messagedata);
	}

	return(1);
}

static gboolean gp2p_mesg_in(GaimAccount *account, char **sender, char **buffer, int *flags, void *data) {
	int gp_ret = 0;

	gp_ret = gp2p_handle_mesg(*sender, *buffer);

	if (gp_ret) {
		return(TRUE);
	}

	return(FALSE);
}

static gboolean plugin_load(GaimPlugin *plugin) {
	void *conv_handle = NULL;

	if (gp2p_loaded == 1) {
		return(TRUE);
	}

	gp2p_loaded = 1;

	conv_handle = gaim_conversations_get_handle();

        gaim_signal_connect(conv_handle, "receiving-im-msg", plugin, GAIM_CALLBACK(gp2p_mesg_in), NULL);

	return(TRUE);
}

static gboolean plugin_unload(GaimPlugin *plugin) {
	void *conv_handle = NULL;

	if (gp2p_loaded == 0) {
		return(TRUE);
	}

	conv_handle = gaim_conversations_get_handle();

	gaim_signals_disconnect_by_handle(conv_handle);

	gp2p_loaded = 0;

	return(TRUE);
}

static void gui_save(void) {
	printf("save!\n");
	return;
}

static GtkWidget *get_config_frame(GaimPlugin *plugin) {
        GtkWidget *ret = NULL;
        GtkWidget *button = NULL;

	/* Container. */
        ret = gtk_vbox_new(FALSE, 18);
	if (ret == NULL) {
		return(NULL);
	}

	gtk_container_set_border_width(GTK_CONTAINER (ret), 12);

	/* Save button */
        button = gtk_button_new_from_stock(GTK_STOCK_SAVE);
	if (button == NULL) {
		return(NULL);
	}
        g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gui_save), NULL);
       	gtk_widget_set_sensitive(button, TRUE);
	gtk_box_pack_start(GTK_BOX(ret), button, FALSE, FALSE, 0);
        gtk_widget_show(button);

        gtk_widget_show_all(ret);
        return(ret);
}


/* Register components with gAIM: */
static GaimGtkPluginUiInfo ui_info = {
	get_config_frame
};

static GaimPluginInfo info = {
	GAIM_PLUGIN_MAGIC,
	GAIM_MAJOR_VERSION,
	GAIM_MINOR_VERSION,
	GAIM_PLUGIN_STANDARD,	                            /**< type           */
	GAIM_GTK_PLUGIN_TYPE,	                            /**< ui_requirement */
	0,	                                            /**< flags          */
	NULL,	                                            /**< dependencies   */
	GAIM_PRIORITY_DEFAULT,	                            /**< priority       */
	GP2P_PLUGIN_ID,	                                    /**< id             */
	("gAIM Peer-to-Peer"),	                    /**< name           */
	GP2P_VERSION,	                                    /**< version        */
	("File-sharing module."),                         /**  summary        */
	("gAIM Peer-to-Peer file sharing allows you to share files amongst your friends."),  /**  description    */
	"Roy Keene <gp2p@rkeene.org>",	                    /**< author         */
	GP2P_WEBSITE,	                                    /**< homepage       */
	plugin_load,	                                    /**< load           */
	plugin_unload,	                                    /**< unload         */
	NULL,	                                            /**< destroy        */
	&ui_info,	                                    /**< ui_info        */
	NULL,	                                            /**< extra_info     */
	NULL,
	NULL
};

static void init_plugin(GaimPlugin *plugin) {
	return;
}

GAIM_INIT_PLUGIN(gp2p, init_plugin, info)
