#include "compat.h"

#include <string.h>
#include <ctype.h>

#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif

#define FRIENDLY_TYPE_CMD 275268
#define FRIENDLY_TYPE_RET 340308

#define FRIENDLY_CMD_SEARCH    0xc50530c8
#define FRIENDLY_CMD_REGISTER  0xc94d5152

static uint32_t valhash(const char *val) {
	uint32_t ret = 0;
	const char *tmp = NULL;

	for (tmp = val; *tmp != '\0'; tmp++) {
		ret <<= 6;
		ret |= toupper(*tmp);
	}

	return(ret);
}

static int parmcpy(char *dest, const char *src, const char *parm, const char *highptr, size_t destlen) {
	size_t tmplen = 0;
	char *tmp = NULL;

	/* Locate the CMD parameter. */
	tmp = strstr(src, parm);
	if (tmp == NULL || tmp > highptr) {
		return(-1);
	}
	tmp = tmp + strlen(parm); /* Skip past the `PARM=' part. */

	/* Ignore quotes. */
	if (*tmp == '\"') {
		tmp++;
	} 

	tmplen = strcspn(tmp, "\t\n\" ]>");
	if (tmplen >= destlen) {
		tmplen = destlen - 1;
	}
	strncpy(dest, tmp, tmplen);

	dest[destlen - 1] = '\0';

	return(0);
}

static char *friendly_handle_cmd_register(char *data, const char *msgid, uint16_t ttl, const char *host) {
	printf("P2P: Executing register: [ttl=%i, host=%s, id=%s] with parms: %s\n", (int) ttl, host, msgid, data);

	return(NULL);
}

static char *friendly_handle_cmd_search(char *data, const char *msgid, uint16_t ttl, const char *host) {
	printf("P2P: Executing search: [ttl=%i, host=%s, id=%s] with parms: %s\n", (int) ttl, host, msgid, data);

	return(NULL);
}

/*
   <CMD ttl=INT host=HOST:PORT cmd=COMMAND ...>...</CMD>
 */
static char *friendly_handle_cmd(char *data, const char *msgid) {
	uint32_t cmdid = 0;
	uint16_t ttl = 0;
	char cmd[128] = {0}, host[128] = {0}, ttlstr[32] = {0};
	char *lowptr = NULL, *highptr = NULL, *message = NULL;
	char *msgret = NULL;
	int pc_ret = -1;

	lowptr = strstr(data, "[CMD ");
	if (lowptr == NULL) {
		return(NULL);
	}

	highptr = strstr(lowptr, "[/CMD]");
	if (highptr == NULL) {
		return(NULL);
	}
	*highptr = '\0';

	highptr = strchr(lowptr, ']');
	if (highptr == NULL) {
		return(NULL);
	}

	/* CMD field. */
	pc_ret = parmcpy(cmd, data, "cmd=", highptr, sizeof(cmd));
	if (pc_ret < 0) {
		return(NULL);
	}

	/* TTL field. */
	pc_ret = parmcpy(ttlstr, data, "ttl=", highptr, sizeof(ttlstr));
	if (pc_ret < 0) {
		return(NULL);
	}
	ttl = atoi(ttlstr);
	if (ttl == 0 || ttl > 20) {
		return(NULL);
	}

	/* HOST field. */
	pc_ret = parmcpy(host, data, "host=", highptr, sizeof(host));
	if (pc_ret < 0) {
		return(NULL);
	}

	message = highptr;
	if (message == NULL) {
		return(NULL);
	}
	message++;

	cmdid = valhash(cmd);

	switch (cmdid) {
		case FRIENDLY_CMD_SEARCH:
			msgret = friendly_handle_cmd_search(message, msgid, ttl, host);
			break;
		case FRIENDLY_CMD_REGISTER:
			msgret = friendly_handle_cmd_register(message, msgid, ttl, host);
			break;
		default:
			printf("P2P: Executing command: %s (0x%x) [ttl=%i, host=%s] with parms: %s\n", cmd, cmdid, ttl, host, message);
			break;
	}

	return(msgret);
}

/*

   <P2P type="cmd" id="abcdef">...</P2P>

 */
int friendly_handle_mesg(const char *source, char *message) {
	uint32_t typeid = 0;
	char *messagedata = NULL, *highptr = NULL;
	char type[128] = {0}, msgid[128] = {0};
	char *tmp = NULL;
	char *cmdret = NULL;
	int pc_ret = -1;

//	printf("Check Message from %s: %s\n", source, message);

	/* Ensure that we have some valid data. */
	if (message == NULL || source == NULL) {
		return(0);
	}

	/* We only handle our type of messages. */
	if (strncmp("[P2P ", message, 5) != 0) {
		return(0);
	}
	tmp = strstr(message, "[/P2P]");
	if (tmp == NULL) {
		return(0);
	}
	*tmp = '\0';

	highptr = strchr(message, ']');
	if (highptr == NULL) {
		return(1);
	}

	/* TYPE field. */
	pc_ret = parmcpy(type, message, "type=", highptr, sizeof(type));
	if (pc_ret < 0) {
		return(1);
	}

	/* ID field. */
	pc_ret = parmcpy(msgid, message, "id=", highptr, sizeof(msgid));
	if (pc_ret < 0) {
		return(1);
	}

	messagedata = highptr;
	if (messagedata == NULL) {
		return(1);
	}

	messagedata++;

	typeid = valhash(type);
	switch (typeid) {
		case FRIENDLY_TYPE_CMD:
			/* What do we do with the return value? XXX */
			cmdret = friendly_handle_cmd(messagedata, msgid);
			break;
		case FRIENDLY_TYPE_RET:
		default:
			printf("P2P Message from %s (type=%s, 0x%x): %s\n", source, type, typeid, messagedata);
	}

	return(1);
}
