   "#! /usr/bin/env tclsh\n"
   "\n"
   "# Copyright (c) 2014-2017 Roy Keene\n"
   "# \n"
   "# Permission is hereby granted, free of charge, to any person obtaining a \n"
   "# copy of this software and associated documentation files (the \"Software\"), \n"
   "# to deal in the Software without restriction, including without limitation \n"
   "# the rights to use, copy, modify, merge, publish, distribute, sublicense, \n"
   "# and/or sell copies of the Software, and to permit persons to whom the \n"
   "# Software is furnished to do so, subject to the following conditions:\n"
   "# \n"
   "# The above copyright notice and this permission notice shall be included in \n"
   "# all copies or substantial portions of the Software.\n"
   "# \n"
   "# THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR \n"
   "# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, \n"
   "# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL \n"
   "# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER \n"
   "# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING \n"
   "# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER \n"
   "# DEALINGS IN THE SOFTWARE.\n"
   "\n"
   "namespace eval ::tuapi {}\n"
   "namespace eval ::tuapi::helper {}\n"
   "namespace eval ::tuapi::cache {}\n"
   "\n"
   "set ::tuapi::_mount_flags(bind) BIND\n"
   "set ::tuapi::_mount_flags(move) MOVE\n"
   "set ::tuapi::_mount_flags(remount) REMOUNT\n"
   "set ::tuapi::_mount_flags(mandlock) MANDLOCK\n"
   "set ::tuapi::_mount_flags(dirsync) DIRSYNC\n"
   "set ::tuapi::_mount_flags(noatime) NOATIME\n"
   "set ::tuapi::_mount_flags(nodiratime) NODIRATIME\n"
   "set ::tuapi::_mount_flags(relatime) RELATIME\n"
   "set ::tuapi::_mount_flags(strictatime) STRICTATIME\n"
   "set ::tuapi::_mount_flags(nodev) NODEV\n"
   "set ::tuapi::_mount_flags(noexec) NOEXEC\n"
   "set ::tuapi::_mount_flags(nosuid) NOSUID\n"
   "set ::tuapi::_mount_flags(ro) RDONLY\n"
   "set ::tuapi::_mount_flags(silent) SILENT\n"
   "set ::tuapi::_mount_flags(synchronous) SYNCHRONOUS\n"
   "set ::tuapi::_mount_flags(sync) SYNCHRONOUS\n"
   "\n"
   "\n"
   "# Determine where to mount a given device (usually by checking \"/etc/fstab\")\n"
   "proc ::tuapi::helper::find_mountpoint {device} {\n"
   "	set data \"\"\n"
   "	catch {\n"
   "		set fd [open \"/etc/fstab\"]\n"
   "		set data [read -nonewline $fd]\n"
   "		close $fd\n"
   "	}\n"
   "\n"
   "	foreach line [split $data \"\\n\"] {\n"
   "		set line [string trim [regsub {#.*$} $line \"\"]]\n"
   "		set line [regsub -all {[ \\t][ \\t][ \\t]*} $line \" \"]\n"
   "\n"
   "		set work [split $line]\n"
   "\n"
   "		set curr_device     [lindex $work 0]\n"
   "		set curr_mountpoint [lindex $work 1]\n"
   "		set curr_fstype     [lindex $work 2]\n"
   "		set curr_opts       [split [lindex $work 3] \",\"]\n"
   "		set curr_dumpfreq   [lindex $work 4]\n"
   "		set curr_fsckpass   [lindex $work 5]\n"
   "\n"
   "\n"
   "		if {$curr_device == $device || $curr_mountpoint == $device} {\n"
   "			return [list source $curr_device target $curr_mountpoint fstype $curr_fstype options $curr_opts dumpfreq $curr_dumpfreq fsckpass $curr_fsckpass]\n"
   "		}\n"
   "	}\n"
   "\n"
   "	return -code error \"no entry found in \\\"/etc/fstab\\\" for \\\"$device\\\"\"\n"
   "}\n"
   "\n"
   "proc ::tuapi::mount args {\n"
   "	set options_list [list]\n"
   "\n"
   "	for {set idx 0} {$idx < [llength $args]} {incr idx} {\n"
   "		set curr_arg [lindex $args $idx]\n"
   "\n"
   "		switch -glob -- $curr_arg {\n"
   "			\"-t\" {\n"
   "				incr idx\n"
   "				set fstype [lindex $args $idx]\n"
   "			}\n"
   "			\"-r\" {\n"
   "				lappend options_list \"RDONLY\"\n"
   "			}\n"
   "			\"-w\" {\n"
   "				set idx [lsearch -exact $options_list \"RDONLY\"]\n"
   "				if {$idx != -1} {\n"
   "					set options_list [lreplace $options_list $idx $idx]\n"
   "				}\n"
   "			}\n"
   "			\"-o\" {\n"
   "				incr idx\n"
   "				set options [lindex $args $idx]\n"
   "			}\n"
   "			\"--\" {\n"
   "				incr idx\n"
   "\n"
   "				break\n"
   "			}\n"
   "			\"-*\" {\n"
   "				return -code error \"unknown option \\\"$curr_arg\\\"\"\n"
   "			}\n"
   "			default {\n"
   "				break\n"
   "			}\n"
   "		}\n"
   "	}\n"
   "\n"
   "	set args [lrange $args $idx end]\n"
   "\n"
   "	if {[llength $args] < 1 || [llength $args] > 2} {\n"
   "		return -code error \"wrong # args: should be \\\"::tuapi::mount ?options? source ?target?\\\"\"\n"
   "	}\n"
   "\n"
   "	set source [lindex $args 0]\n"
   "\n"
   "	if {[llength $args] == 2} {\n"
   "		set target [lindex $args 1]\n"
   "	} else {\n"
   "		array set mountinfo [::tuapi::helper::find_mountpoint $source]\n"
   "		set source $mountinfo(source)\n"
   "		set target $mountinfo(target)\n"
   "\n"
   "		if {![info exists fstype]} {\n"
   "			set fstype $mountinfo(fstype)\n"
   "		}\n"
   "\n"
   "		if {![info exists options]} {\n"
   "			set options $mountinfo(options)\n"
   "		}\n"
   "	}\n"
   "\n"
   "	# Ensure all mount-related parameters have been computed\n"
   "	if {![info exists fstype]} {\n"
   "		set fstype \"auto\"\n"
   "	}\n"
   "\n"
   "	if {![info exists options]} {\n"
   "		set options [list]\n"
   "	}\n"
   "\n"
   "	# Process options\n"
   "	foreach option $options {\n"
   "		set option_lc [string tolower $option]\n"
   "\n"
   "		# Special option handling\n"
   "		switch -- $option_lc {\n"
   "			\"defaults\" {\n"
   "				set options_list [list]\n"
   "				unset -nocomplain unknown_options\n"
   "\n"
   "				continue\n"
   "			}\n"
   "			\"rw\" {\n"
   "				set option_lc \"noro\"\n"
   "			}\n"
   "			\"norw\" {\n"
   "				set option_lc \"ro\"\n"
   "			}\n"
   "		}\n"
   "\n"
   "		# Example: noatime\n"
   "		if {[info exists ::tuapi::_mount_flags($option_lc)]} {\n"
   "			lappend options_list $::tuapi::_mount_flags($option_lc)\n"
   "\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		# Example: atime\n"
   "		if {[info exists ::tuapi::_mount_flags(no$option_lc)]} {\n"
   "			set idx [lsearch -exact $options_list $::tuapi::_mount_flags(no$option_lc)]\n"
   "			if {$idx != -1} {\n"
   "				set options_list [lreplace $options_list $idx $idx]\n"
   "			}\n"
   "\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		# Example: norelatime\n"
   "		if {[string match \"no*\" $option_lc]} {\n"
   "			set neg_option_lc [string range $option_lc 2 end]\n"
   "\n"
   "			if {[info exists ::tuapi::_mount_flags($neg_option_lc)]} {\n"
   "				set idx [lsearch -exact $options_list $::tuapi::_mount_flags($neg_option_lc)]\n"
   "				if {$idx != -1} {\n"
   "					set options_list [lreplace $options_list $idx $idx]\n"
   "				}\n"
   "\n"
   "				continue\n"
   "			}\n"
   "		}\n"
   "\n"
   "		# Accumulate unknown options\n"
   "		lappend unknown_options $option\n"
   "	}\n"
   "\n"
   "	# Use \"swapon\" if this is swap\n"
   "	if {$fstype == \"swap\"} {\n"
   "		return [::tuapi::syscall::swapon $source]\n"
   "	}\n"
   "\n"
   "	# Otherwise, call \"mount\" system call\n"
   "	## If we have accumulated any unknown options, pass them as a\n"
   "	## comma-seperated value string\n"
   "	if {[info exists unknown_options]} {\n"
   "		set data [join $unknown_options \",\"]\n"
   "\n"
   "		return [::tuapi::syscall::mount $source $target $fstype $options_list $data]\n"
   "	}\n"
   "\n"
   "	return [::tuapi::syscall::mount $source $target $fstype $options_list]\n"
   "}\n"
   "\n"
   "proc ::tuapi::umount {dir {flags \"\"}} {\n"
   "	return [::tuapi::syscall::umount $dir [string toupper $flags]]\n"
   "}\n"
   "\n"
   "proc ::tuapi::kill {pid sig} {\n"
   "	return [::tuapi::syscall::kill $pid [string toupper $sig]]\n"
   "}\n"
   "\n"
   "proc ::tuapi::killpg {pgroup sig} {\n"
   "	if {$pgroup <= 1} {\n"
   "		return -code error \"invalid process group specified (must be greater than 1)\"\n"
   "	}\n"
   "\n"
   "	return [::tuapi::syscall::kill -$pgroup [string toupper $sig]]\n"
   "}\n"
   "\n"
   "proc ::tuapi::ifconfig args {\n"
   "	if {[llength $args] == 0} {\n"
   "		# Return information on all interfaces\n"
   "		set retlist [list]\n"
   "		foreach interface [::tuapi::syscall::ifconfig] {\n"
   "			lappend retlist $interface [::tuapi::syscall::ifconfig $interface]\n"
   "		}\n"
   "\n"
   "		return $retlist\n"
   "	}\n"
   "\n"
   "	set interface [lindex $args 0]\n"
   "	set args [lrange $args 1 end]\n"
   "\n"
   "	array set ifaceinfo [::tuapi::syscall::ifconfig $interface]\n"
   "\n"
   "	if {[llength $args] == 0} {\n"
   "		return [array get ifaceinfo]\n"
   "	}\n"
   "\n"
   "	for {set idx 0} {$idx < [llength $args]} {incr idx} {\n"
   "		set opt [lindex $args $idx]\n"
   "\n"
   "		switch -- $opt {\n"
   "			\"up\" {\n"
   "				if {[info exists ifaceinfo(flags)]} {\n"
   "					set flags $ifaceinfo(flags)\n"
   "				} else {\n"
   "					set flags \"\"\n"
   "				}\n"
   "\n"
   "				foreach newflag [list UP] {\n"
   "					if {[lsearch -exact $flags $newflag] == -1} {\n"
   "						lappend flags $newflag\n"
   "					}\n"
   "				}\n"
   "\n"
   "				::tuapi::syscall::ifconfig $interface flags $flags\n"
   "			}\n"
   "			\"down\" {\n"
   "				if {[info exists ifaceinfo(flags)]} {\n"
   "					set flags $ifaceinfo(flags)\n"
   "				} else {\n"
   "					set flags \"\"\n"
   "				}\n"
   "\n"
   "				set flagidx [lsearch -exact $flags \"UP\"]\n"
   "				if {$flagidx != -1} {\n"
   "					set flags [lreplace $flags $flagidx $flagidx]\n"
   "\n"
   "					::tuapi::syscall::ifconfig $interface flags $flags\n"
   "				}\n"
   "			}\n"
   "			default {\n"
   "				incr idx\n"
   "				set optval [lindex $args $idx]\n"
   "\n"
   "				::tuapi::syscall::ifconfig $interface $opt $optval\n"
   "			}\n"
   "		}\n"
   "	}\n"
   "}\n"
   "\n"
   "proc ::tuapi::helper::foreach_line {fd sep code} {\n"
   "	while {![eof $fd]} {\n"
   "		gets $fd line\n"
   "\n"
   "		regsub { *#.*$} $line {} line\n"
   "\n"
   "		if {$line == \"\"} {\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		set line [split $line $sep]\n"
   "\n"
   "		uplevel 1 [list set line $line]\n"
   "		uplevel 1 $code\n"
   "	}\n"
   "	uplevel 1 [list unset -nocomplain line]\n"
   "}\n"
   "\n"
   "proc ::tuapi::helper::recursive_glob {path pattern code {depth 1}} {\n"
   "	foreach filename [glob -nocomplain -directory $path -type f $pattern] {\n"
   "		uplevel $depth [list set filename $filename]\n"
   "		uplevel $depth $code\n"
   "	}\n"
   "\n"
   "	incr depth\n"
   "\n"
   "	foreach dirname [glob -nocomplain -directory $path -type d *] {\n"
   "		set dirinfo(type) unknown\n"
   "		catch {\n"
   "			file lstat $dirname dirinfo\n"
   "		}\n"
   "\n"
   "		if {$dirinfo(type) == \"link\"} {\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		::tuapi::helper::recursive_glob $dirname $pattern $code $depth\n"
   "	}\n"
   "}\n"
   "\n"
   "proc ::tuapi::modprobe args {\n"
   "	# Process arguments\n"
   "	set options(call_insmod) 1\n"
   "	set idx 0\n"
   "	set nextIsArgs 0\n"
   "	foreach arg $args {\n"
   "		if {$nextIsArgs} {\n"
   "			set options(args) $arg\n"
   "\n"
   "			set nextIsArgs 0\n"
   "\n"
   "			incr idx\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		switch -- $arg {\n"
   "			\"-dontload\" {\n"
   "				set options(call_insmod) 0\n"
   "			}\n"
   "			\"-args\" {\n"
   "				set nextIsArgs 1\n"
   "			}\n"
   "			\"--\" {\n"
   "				incr idx\n"
   "				break\n"
   "			}\n"
   "			default {\n"
   "				break\n"
   "			}\n"
   "		}\n"
   "\n"
   "		incr idx\n"
   "	}\n"
   "	set args [lrange $args $idx end]\n"
   "\n"
   "	# Set initial retval\n"
   "	set retval [list]\n"
   "\n"
   "	# Set module base directory\n"
   "	set modules_dir [file join /lib/modules $::tcl_platform(osVersion)]\n"
   "\n"
   "	# Load device names\n"
   "	if {![info exists ::tuapi::cache::alias2module]} {\n"
   "		set devnames_file [file join $modules_dir modules.devname]\n"
   "		set fd [open $devnames_file]\n"
   "		::tuapi::helper::foreach_line $fd \" \" {\n"
   "			set module [lindex $line 0]\n"
   "			set device [lindex $line 1]\n"
   "			set id [lindex $line 2]\n"
   "\n"
   "			set id_type [string index $id 0]\n"
   "			set id_type [string map [list \"c\" \"char\" \"b\" \"block\"] $id_type]\n"
   "			set id [split [string range $id 1 end] :]\n"
   "			set id_alias \"${id_type}-major-[lindex $id 0]-[lindex $id 1]\"\n"
   "\n"
   "			set \"alias2module(/dev/${device})\" $module\n"
   "			set alias2module($id_alias) $module\n"
   "		}\n"
   "		close $fd\n"
   "\n"
   "		# Load aliases\n"
   "		set aliases_file [file join $modules_dir modules.alias]\n"
   "		set fd [open $aliases_file]\n"
   "		::tuapi::helper::foreach_line $fd \" \" {\n"
   "			set alias [lindex $line 1]\n"
   "			set module [lindex $line 2]\n"
   "\n"
   "			set alias2module($alias) $module\n"
   "			if {[string match {*\\**} $alias]} {\n"
   "				set alias2module_wildcards($alias) $module\n"
   "			}\n"
   "		}\n"
   "		close $fd\n"
   "\n"
   "		# Load dependencies\n"
   "		set deps_file [file join $modules_dir modules.dep]\n"
   "		set fd [open $deps_file]\n"
   "		::tuapi::helper::foreach_line $fd \":\" {\n"
   "			set module [string trim [lindex $line 0]]\n"
   "			set deps [lreverse [split [string trim [join [lrange $line 1 end]]]]]\n"
   "\n"
   "			set module_basename [file rootname [file tail $module]]\n"
   "			set module_basename_alt1 [string map [list \"_\" \"-\"] $module_basename]\n"
   "			set module_basename_alt2 [string map [list \"-\" \"_\"] $module_basename]\n"
   "\n"
   "			set alias2module($module_basename) $module\n"
   "			set alias2module($module_basename_alt1) $module\n"
   "			set alias2module($module_basename_alt2) $module\n"
   "\n"
   "			if {[llength $deps] != 0} {\n"
   "				set module2deps($module) $deps\n"
   "			}\n"
   "		}\n"
   "		close $fd\n"
   "\n"
   "		set ::tuapi::cache::alias2module [array get alias2module]\n"
   "		set ::tuapi::cache::alias2module_wildcards [array get alias2module_wildcards]\n"
   "		set ::tuapi::cache::module2deps [array get module2deps]\n"
   "	} else {\n"
   "		array set alias2module $::tuapi::cache::alias2module\n"
   "		array set alias2module_wildcards $::tuapi::cache::alias2module_wildcards\n"
   "		array set module2deps $::::tuapi::cache::module2deps\n"
   "	}\n"
   "\n"
   "	# Determine list of modules\n"
   "	set all_modules [list]\n"
   "	foreach modules $args {\n"
   "		foreach module $modules {\n"
   "			lappend all_modules $module\n"
   "		}\n"
   "	}\n"
   "\n"
   "	# Determine what modules to add the arguments to\n"
   "	if {[info exists options(args)]} {\n"
   "		foreach arg [split $options(args) \" \\n\\t\"] {\n"
   "			set arg [string trim $arg]\n"
   "			if {$arg == \"\"} {\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			if {[string match \"*=*\" $arg]} {\n"
   "				set work [split $arg =]\n"
   "\n"
   "				set name [lindex $work 0]\n"
   "				set value [join [lrange $work 1 end] =]\n"
   "			} else {\n"
   "				set name $arg\n"
   "				unset -nocomplain value\n"
   "			}\n"
   "\n"
   "			if {[string match \"*.*\" $name]} {\n"
   "				set work [split $name .]\n"
   "\n"
   "				set module [lindex $work 0]\n"
   "				if {$module == \"\"} {\n"
   "					set modules [list]\n"
   "				} else {\n"
   "					set modules [list $module]\n"
   "				}\n"
   "\n"
   "				set name [join [lrange $work 1 end] .]\n"
   "			} else {\n"
   "				set modules [list]\n"
   "			}\n"
   "\n"
   "			if {[llength $modules] == 0} {\n"
   "				set modules $all_modules\n"
   "			}\n"
   "\n"
   "			foreach module $modules {\n"
   "				if {[info exists value]} {\n"
   "					append modules_args($module) \"$name=$value \"\n"
   "				} else {\n"
   "					append modules_args($module) \"$name \"\n"
   "				}\n"
   "			}\n"
   "		}\n"
   "	}\n"
   "\n"
   "	# Load modules\n"
   "	foreach module $all_modules {\n"
   "		# If the module is given as an absolute path, ignore the path\n"
   "		# and process just as we would if the name were given alone\n"
   "		# This may be wrong, but otherwise dependency matching would\n"
   "		# be harder\n"
   "		if {[string index $module 0] == \"/\" && [file exists $module]} {\n"
   "			set module [file rootname [file tail $module]]\n"
   "		}\n"
   "\n"
   "		for {set try 0} {$try < 100} {incr try} {\n"
   "			if {![info exists alias2module($module)]} {\n"
   "				# If no exact match found, process wildcard entries\n"
   "				set found_wildcard_match 0\n"
   "				set tmp_matched_modules [list]\n"
   "				foreach alias [array name alias2module_wildcards] {\n"
   "					if {[string match $alias $module]} {\n"
   "						set module $alias2module_wildcards($alias)\n"
   "\n"
   "						lappend tmp_matched_modules $module\n"
   "\n"
   "						incr found_wildcard_match 1\n"
   "					}\n"
   "				}\n"
   "\n"
   "				if {!$found_wildcard_match} {\n"
   "					break\n"
   "				}\n"
   "\n"
   "				if {$found_wildcard_match > 1} {\n"
   "					# Multiple matches, try to pick the best one\n"
   "					foreach tmp_module $tmp_matched_modules {\n"
   "						# First, prefer things that do not contain generic\n"
   "						if {![string match \"*generic*\" $tmp_module]} {\n"
   "							set module $tmp_module\n"
   "						}\n"
   "					}\n"
   "				}\n"
   "			}\n"
   "\n"
   "			set module $alias2module($module)\n"
   "		}\n"
   "\n"
   "		if {[info exists module2deps($module)]} {\n"
   "			set load $module2deps($module)\n"
   "		} else {\n"
   "			set load [list]\n"
   "		}\n"
   "\n"
   "		lappend load $module\n"
   "\n"
   "		foreach module $load {\n"
   "			if {[string match \"/dev/*\" $module]} {\n"
   "				return -code error \"Unable to lookup device node module for $module\"\n"
   "			}\n"
   "\n"
   "			set module [file join $modules_dir $module]\n"
   "\n"
   "			unset -nocomplain module_args\n"
   "			set module_short [file rootname [file tail $module]]\n"
   "			if {[info exists modules_args($module_short)]} {\n"
   "				set module_args [string trim $modules_args($module_short)]\n"
   "			}\n"
   "\n"
   "			if {$options(call_insmod)} {\n"
   "				if {[catch {\n"
   "					if {[info exists module_args]} {\n"
   "						::tuapi::syscall::insmod $module $module_args\n"
   "					} else {\n"
   "						::tuapi::syscall::insmod $module\n"
   "					}\n"
   "				}]} {\n"
   "					continue\n"
   "				}\n"
   "			}\n"
   "\n"
   "			lappend retval $module\n"
   "		}\n"
   "	}\n"
   "\n"
   "	return $retval\n"
   "}\n"
   "\n"
   "# Scan the various buses attached to the system and load the appropriate\n"
   "# kernel modules\n"
   "proc ::tuapi::scan_and_load_kernel_modules args {\n"
   "	set parameters [list \"ata_generic.all_generic_ide=1\"]\n"
   "	foreach arg $args {\n"
   "		if {[info exists var_to_set]} {\n"
   "			set $var_to_set $arg\n"
   "\n"
   "			unset var_to_set\n"
   "\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		if {[info exists var_to_lappend]} {\n"
   "			lappend $var_to_lappend $arg\n"
   "\n"
   "			unset var_to_lappend\n"
   "\n"
   "			continue\n"
   "		}\n"
   "\n"
   "		switch -- $arg {\n"
   "			\"-arg\" {\n"
   "				set var_to_lappend parameters\n"
   "			}\n"
   "			\"-args\" {\n"
   "				set var_to_set parameters\n"
   "			}\n"
   "			default {\n"
   "				return -code error \"Unknown option: $arg\"\n"
   "			}\n"
   "		}\n"
   "	}\n"
   "\n"
   "	set modules [list]\n"
   "\n"
   "	# Determine which modules are already loaded\n"
   "	foreach module [glob -tails -nocomplain -directory /sys/module -type d *] {\n"
   "		set alt_module1 [string map [list \"_\" \"-\"] $module]\n"
   "		set alt_module2 [string map [list \"-\" \"_\"] $module]\n"
   "\n"
   "		set loaded($module) 1\n"
   "		set loaded($alt_module1) 1\n"
   "		set loaded($alt_module2) 1\n"
   "	}\n"
   "\n"
   "	::tuapi::helper::recursive_glob /sys/devices modalias {\n"
   "		set fd [open $filename r]\n"
   "		::tuapi::helper::foreach_line $fd \"\\n\" {\n"
   "			foreach module [::tuapi::modprobe -dontload -- $line] {\n"
   "				if {[lsearch -exact $modules $module] != -1} {\n"
   "					continue\n"
   "				}\n"
   "\n"
   "				if {![file exists $module]} {\n"
   "					continue\n"
   "				}\n"
   "\n"
   "				set rootname [file rootname [file tail $module]]\n"
   "				if {[info exists loaded($rootname)]} {\n"
   "					continue\n"
   "				}\n"
   "\n"
   "				lappend modules $module\n"
   "			}\n"
   "		}\n"
   "		close $fd\n"
   "	}\n"
   "\n"
   "	set failed_to_load [list]\n"
   "	set able_to_load [list]\n"
   "	foreach module $modules {\n"
   "		if {[::tuapi::modprobe -args $parameters $module] == \"\"} {\n"
   "			lappend failed_to_load $module\n"
   "		} else {\n"
   "			lappend able_to_load $module\n"
   "		}\n"
   "	}\n"
   "\n"
   "	return [list -failed $failed_to_load -loaded $able_to_load]\n"
   "}\n"
   "\n"
   "# Create UNIX-like procs meant to be used interactively\n"
   "proc ::tuapi::create_unix_commands {} {\n"
   "	proc ::cat args {\n"
   "		foreach file $args {\n"
   "			if {[catch {\n"
   "				set fd [open $file]\n"
   "			} err]} {\n"
   "				puts stderr \"Unable to open \\\"$file\\\": $err\"\n"
   "\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			fcopy $fd stdout\n"
   "			close $fd\n"
   "		}\n"
   "	}\n"
   "\n"
   "	proc ::ls args {\n"
   "		set options(long) 0\n"
   "		set options(one) 0\n"
   "		set options(skipdot) 1\n"
   "		set options(norecurseintotopleveldirs) 0\n"
   "\n"
   "		set idx 0\n"
   "		foreach arg $args {\n"
   "			if {[string match \"-*\" $arg]} {\n"
   "				set args [lreplace $args $idx $idx]\n"
   "				if {$arg == \"--\"} {\n"
   "					break\n"
   "				}\n"
   "\n"
   "				if {[string range $arg 0 1] == \"--\"} {\n"
   "					set opts [list [string range $arg 2 end]]\n"
   "				} else {\n"
   "					set opts [split [string range $arg 1 end] \"\"]\n"
   "				}\n"
   "\n"
   "				foreach opt $opts {\n"
   "					switch -- $opt {\n"
   "						\"l\" {\n"
   "							set options(long) 1\n"
   "							set options(one) 0\n"
   "						}\n"
   "						\"1\" {\n"
   "							set options(one) 1\n"
   "							set options(long) 0\n"
   "						}\n"
   "						\"d\" {\n"
   "							set options(norecurseintotopleveldirs) 1\n"
   "						}\n"
   "						\"a\" {\n"
   "							set options(skipdot) 0\n"
   "						}\n"
   "					}\n"
   "				}\n"
   "\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			incr idx\n"
   "		}\n"
   "\n"
   "		if {[llength $args] == 0} {\n"
   "			set args [list \".\"]\n"
   "		}\n"
   "\n"
   "		set nodes [list]\n"
   "		foreach arg $args {\n"
   "			unset -nocomplain fileinfo\n"
   "			catch {\n"
   "				file stat $arg fileinfo\n"
   "			}\n"
   "\n"
   "			if {![info exists fileinfo]} {\n"
   "				puts stderr \"No such file or directory: $arg\"\n"
   "\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			if {$fileinfo(type) == \"directory\"} {\n"
   "				if {$options(norecurseintotopleveldirs)} {\n"
   "					lappend nodes $arg\n"
   "				} else {\n"
   "					lappend nodes {*}[glob -nocomplain -directory $arg -tails *]\n"
   "				}\n"
   "			} else {\n"
   "				lappend nodes $arg\n"
   "			}\n"
   "\n"
   "		}\n"
   "\n"
   "		set newline_required 0\n"
   "		foreach node $nodes {\n"
   "			unset -nocomplain fileinfo\n"
   "\n"
   "			if {$options(one)} {\n"
   "				puts $node\n"
   "			} elseif {$options(long)} {\n"
   "				catch {\n"
   "					file stat $node fileinfo\n"
   "				}\n"
   "\n"
   "				if {![info exists fileinfo]} {\n"
   "					array set fileinfo [list mode 0 nlink 0 uid -1 gid -1 size 0 mtime 0]\n"
   "				}\n"
   "\n"
   "				set date [clock format $fileinfo(mtime) -format {%b %e %H:%M}]\n"
   "\n"
   "				switch -- $fileinfo(type) {\n"
   "					\"directory\" {\n"
   "						set typeid \"d\"\n"
   "					}\n"
   "					\"blockSpecial\" {\n"
   "						set typeid \"b\"\n"
   "					}\n"
   "					\"characterSpecial\" {\n"
   "						set typeid \"c\"\n"
   "					}\n"
   "					\"file\" {\n"
   "						set typeid \"-\"\n"
   "					}\n"
   "					\"socket\" {\n"
   "						set typeid \"s\"\n"
   "					}\n"
   "					default {\n"
   "						set typeid \"?\"\n"
   "					}\n"
   "				}\n"
   "\n"
   "				puts [format {%s%04o %5s %6s %6s %10s %12s %s} $typeid [expr {$fileinfo(mode) & 07777}] $fileinfo(nlink) $fileinfo(uid) $fileinfo(gid) $fileinfo(size) $date $node]\n"
   "\n"
   "			} else {\n"
   "				puts -nonewline \"$node \"\n"
   "				set newline_required 1\n"
   "			}\n"
   "		}\n"
   "\n"
   "		if {$newline_required} {\n"
   "			puts \"\"\n"
   "		}\n"
   "	}\n"
   "\n"
   "	proc ::modprobe args {\n"
   "		::tuapi::modprobe {*}$args\n"
   "	}\n"
   "\n"
   "	proc ::ps {} {\n"
   "		set format {%-6s %5s %5s %3s %5s %-6s %8s %s}\n"
   "		puts [format $format UID PID PPID C STIME TTY TIME CMD]\n"
   "		foreach pid [lsort -dictionary [glob -nocomplain -directory /proc -tails {[0-9]*}]] {\n"
   "			if {![string is integer $pid]} {\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			set procfile [file join /proc $pid]\n"
   "\n"
   "			unset -nocomplain pidinfo\n"
   "			catch {\n"
   "				file stat $procfile pidinfo\n"
   "			}\n"
   "\n"
   "			if {![info exists pidinfo]} {\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			set pidinfo(pid) $pid\n"
   "			set pidinfo(ppid) ?\n"
   "			set pidinfo(cpuutil) ?\n"
   "			set pidinfo(starttime) ?\n"
   "			set pidinfo(tty) ?\n"
   "			set pidinfo(cputime) ?\n"
   "			set pidinfo(cmd) \"\"\n"
   "\n"
   "			unset -nocomplain fd\n"
   "			catch {\n"
   "				set fd [open [file join $procfile cmdline]]\n"
   "			}\n"
   "			if {[info exists fd]} {\n"
   "				set pidinfo(cmd) [string trim [join [split [read $fd] \"\\0\\n\\r\"]]]\n"
   "				close $fd\n"
   "				unset fd\n"
   "			}\n"
   "			if {![info exists pidinfo(cmd)] || $pidinfo(cmd) == \"\"} {\n"
   "				catch {\n"
   "					set fd [open [file join $procfile comm]]\n"
   "				}\n"
   "				if {[info exists fd]} {\n"
   "					set pidinfo(cmd) \"\\[[string trim [join [split [read $fd] \"\\0\\n\\r\"]]]\\]\"\n"
   "					close $fd\n"
   "				}\n"
   "			}\n"
   "\n"
   "			puts [format $format $pidinfo(uid) $pidinfo(pid) $pidinfo(ppid) $pidinfo(cpuutil) $pidinfo(starttime) $pidinfo(tty) $pidinfo(cputime) $pidinfo(cmd)]\n"
   "		}\n"
   "	}\n"
   "\n"
   "	proc ::dmesg {} {\n"
   "		puts [::tuapi::syscall::klogctl read]\n"
   "	}\n"
   "\n"
   "	proc ::ulimit {limit {val \"\"}} {\n"
   "		set mapping(-c) [list CORE]\n"
   "		set mapping(-d) [list DATA]\n"
   "		set mapping(-e) [list NICE]\n"
   "		set mapping(-f) [list FSIZE]\n"
   "		set mapping(-i) [list SIGPENDING]\n"
   "		set mapping(-l) [list MEMLOCK]\n"
   "		set mapping(-m) [list RSS]\n"
   "		set mapping(-n) [list NOFILE]\n"
   "		set mapping(-q) [list MSGQUEUE]\n"
   "		set mapping(-r) [list RTPRIO]\n"
   "		set mapping(-s) [list STACK]\n"
   "		set mapping(-t) [list CPU]\n"
   "		set mapping(-u) [list NPROC]\n"
   "		set mapping(-v) [list AS]\n"
   "		set mapping(-x) [list LOCKS]\n"
   "		set help(CORE) {core file size          (blocks, -c)}\n"
   "		set help(DATA) {data seg size           (kbytes, -d)}\n"
   "		set help(NICE) {scheduling priority             (-e)}\n"
   "		set help(FSIZE) {file size               (blocks, -f)}\n"
   "		set help(SIGPENDING) {pending signals                 (-i)}\n"
   "		set help(MEMLOCK) {max locked memory       (kbytes, -l)}\n"
   "		set help(RSS) {max memory size         (kbytes, -m)}\n"
   "		set help(NOFILE) {open files                      (-n)}\n"
   "		set help(-p) {pipe size            (512 bytes, -p)}\n"
   "		set help(MSGQUEUE) {POSIX message queues     (bytes, -q)}\n"
   "		set help(RTPRIO) {real-time priority              (-r)}\n"
   "		set help(STACK) {stack size              (kbytes, -s)}\n"
   "		set help(CPU) {cpu time               (seconds, -t)}\n"
   "		set help(NPROC) {max user processes              (-u)}\n"
   "		set help(AS) {virtual memory          (kbytes, -v)}\n"
   "		set help(LOCKS) {file locks                      (-x)}\n"
   "\n"
   "		foreach {limitopt limitoptvals} [array get mapping] {\n"
   "			foreach limitoptval $limitoptvals {\n"
   "				lappend mapping(-a) $limitoptval\n"
   "			}\n"
   "		}\n"
   "\n"
   "		set opts $mapping($limit)\n"
   "\n"
   "		if {[llength $opts] != 1 && $val != \"\"} {\n"
   "			return -code error \"Unable to set multiple limits\"\n"
   "		}\n"
   "\n"
   "		foreach opt $opts {\n"
   "			unset -nocomplain optval\n"
   "\n"
   "			if {$val != \"\"} {\n"
   "				catch {\n"
   "					::tuapi::syscall::rlimit set $opt $val\n"
   "				}\n"
   "				set include_help \"\"\n"
   "			} else {\n"
   "				set include_help \"$help($opt) \"\n"
   "			}\n"
   "\n"
   "			catch {\n"
   "				set optval [::tuapi::syscall::rlimit get $opt]\n"
   "			}\n"
   "\n"
   "			if {![info exists optval]} {\n"
   "				continue\n"
   "			}\n"
   "\n"
   "			puts \"${include_help}$optval\"\n"
   "		}\n"
   "	}\n"
   "\n"
   "	proc ::ifconfig {{ifacelist \"\"} {config \"\"}} {\n"
   "		if {$ifacelist == \"\" || $ifacelist == \"-a\"} {\n"
   "			set ifacelist [tuapi::syscall::ifconfig]\n"
   "			set config \"\"\n"
   "		}\n"
   "\n"
   "		if {$config != \"\"} {\n"
   "			if {[string match \"*.*.*.*\" [lindex $config 0]]} {\n"
   "				set config [linsert $config 0 \"address\"]\n"
   "			}\n"
   "\n"
   "			puts [list ::tuapi::ifconfig [lindex $ifacelist 0] {*}$config]\n"
   "			return [::tuapi::ifconfig [lindex $ifacelist 0] {*}$config]\n"
   "		}\n"
   "\n"
   "		foreach iface $ifacelist {\n"
   "			unset -nocomplain ifaceinfo\n"
   "			array set ifaceinfo [tuapi::syscall::ifconfig $iface]\n"
   "\n"
   "			set secondline \"\"\n"
   "			foreach {label entry} [list inet address netmask netmask broadcast broadcast] {\n"
   "				if {![info exists ifaceinfo($entry)]} {\n"
   "					continue\n"
   "				}\n"
   "\n"
   "				append secondline \" $label $ifaceinfo($entry)\"\n"
   "			}\n"
   "\n"
   "			puts \"$iface: flags=<[join $ifaceinfo(flags) ,]> mtu $ifaceinfo(mtu) index $ifaceinfo(index)\"\n"
   "			puts \"\\t[string trim $secondline]\"\n"
   "			if {[info exists ifaceinfo(hwaddr)]} {\n"
   "				puts \"\\tether $ifaceinfo(hwaddr)\"\n"
   "			}\n"
   "		}\n"
   "	}\n"
   "}\n"
