/*

	Sound interface for Netfone

	Designed and implemented in July of 1990 by John Walker

*/

#ifdef sun

#include <sys/types.h>
#include <sys/dir.h>
#include <sys/file.h>

#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <assert.h>

#include <sys/ioctl.h>
#include <sun/audioio.h>

#define SoundFile       "/dev/audio"
#define AUDIO_CTLDEV    "/dev/audioctl"

#define MAX_GAIN	100

#define TRUE  1
#define FALSE 0

#define V     (void)

struct sound_buf {
    struct sound_buf *snext;	      /* Next sound buffer */
    int sblen;			      /* Length of this sound buffer */
    unsigned char sbtext[2];	      /* Actual sampled sound */
};

/*  Local variables  */

static int audiof = -1; 	      /* Audio device file descriptor */
static int Audio_fd;		      /* Audio control port */
static audio_info_t Audio_info;       /* Current configuration info */
struct sound_buf *sbchain = NULL,     /* Sound buffer chain links */
		 *sbtail = NULL;
static int sbtotal = 0; 	      /* Total sample bytes in memory */
static int recording = FALSE;	      /* Recording in progress ? */
static int playing = FALSE;	      /* Replay in progress ? */
/* static int playqsize;  */		    /* Output queue size */
static int playlen = 0; 	      /* Length left to play */
static unsigned char *playbuf= NULL;  /* Current play pointer */
static int squelch = 0; 	      /* Squelch value */

/* Convert local gain into device parameters */

static unsigned scale_gain(g)
  unsigned g;
{
    return (AUDIO_MIN_GAIN + (unsigned)
	irint(((double) (AUDIO_MAX_GAIN - AUDIO_MIN_GAIN)) *
	((double)g / (double)MAX_GAIN)));
}


/*  SOUNDINIT  --  Open the sound peripheral and initialise for
		   access.  Return TRUE if successful, FALSE
		   otherwise.  */

int soundinit(iomode)
  int iomode;
{
    assert(audiof == -1);
    if ((audiof = open(SoundFile, iomode)) >= 0) {

	if ((Audio_fd = open(AUDIO_CTLDEV, O_RDWR)) < 0) {
	    perror(AUDIO_CTLDEV);
	    return FALSE;
	}
/*fcntl(audiof, F_SETFL, O_NDELAY);*/
	return TRUE;
    }
    return FALSE;
}

/*  SOUNDTERM  --  Close the sound device.  */

void soundterm()
{
	if (audiof >= 0) {
	   V close(audiof);
	   V close(Audio_fd);
	   audiof = -1;
	}
}

#ifdef NEEDED

/*  SOUNDRECEND  --  Stop recording of sound.  */

void soundrecend(reclen, recbuf)
  int *reclen;
  unsigned char **recbuf;
{
    int dummy;

    assert(recording);
    recording = FALSE;
    *reclen = 0;
    *recbuf = NULL;

    /* Put all the individual sound buffers together into one
       Godzilla sound buffer and return it. */

    if (sbtotal > 0) {
	unsigned char *sb = (unsigned char *) malloc(sbtotal);

	*recbuf = sb;
	if (sb != NULL)
	    *reclen = sbtotal;
	sbtotal = 0;
	while (sbchain != NULL) {
	    if (sb != NULL) {
		memcpy(sb, sbchain->sbtext, sbchain->sblen);
		sb += sbchain->sblen;
	    }
	    sbtail = sbchain->snext;
	    free(sbchain);
	    sbchain = sbtail;
	}

	/* Now apply squelch to the start and end of the consolidated
	   buffer if the squelch has been set nonzero. */

	if (squelch > 0) {
	    unsigned char *start = *recbuf,
			  *end = *recbuf + (*reclen - 1);

	    while ((start < end) && (((*start & 0x7F) ^ 0x7F) <= squelch))
		start++;

	    while ((end > start) && (((*end & 0x7F) ^ 0x7F) <= squelch))
		end--;

	    if (start >= end) {
		free(*recbuf);
		*reclen = 0;
		*recbuf = NULL;
/* printf("Entire buffer squelched.\n"); */
	    } else {
/* printf("Squelched %d bytes from start, %d from end.\n",
  start - *recbuf, (*recbuf + (*reclen - 1)) - end); */
		*reclen = (end - start) + 1;
		bcopy(start, *recbuf, *reclen);
	    }
	}
    }
}
#endif

/*  SOUNDRECORD  --  Begin recording of sound.	*/

int soundrecord()
{
    int dummy;

    assert(!recording);
    recording = TRUE;
    return recording;
}

/*  SOUNDPLAY  --  Begin playing a sound.  */

void soundplay(len, buf)
  int len;
  unsigned char *buf;
{
    int ios;

    assert(audiof != -1);
    while (TRUE) {
	ios = write(audiof, buf, len);
	if (ios == -1) {
	    usleep(100000);
	} else {
	    if (ios < len) {
		buf += ios;
		len -= ios;
	    } else {
		break;
	    }
	}
    }
}

/*  SOUNDPLAYVOL  --  Set playback volume from 0 (silence) to 100 (full on). */

void soundplayvol(value)
  int value;
{
    AUDIO_INITINFO(&Audio_info);
    Audio_info.play.gain = scale_gain(value);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set play volume");
    }
}

/*  SOUNDRECGAIN  --  Set recording gain from 0 (minimum) to 100 (maximum).  */

void soundrecgain(value)
  int value;
{
    AUDIO_INITINFO(&Audio_info);
    Audio_info.record.gain = scale_gain(value);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set record gain");
    }
}

#ifdef NEEDED

/*  SOUNDMONGAIN  --  Set monitor (sidetone) gain from 0 (minimum) to 100 (maximum).  */

void soundmongain(value)
  int value;
{
    AUDIO_INITINFO(&Audio_info);
    Audio_info.monitor_gain = scale_gain(value);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set monitor volume");
    }
}

/*  SOUNDSQUELCH  --  Set squelch level for recording.	*/

void soundsquelch(value)
  int value;
{
    squelch = value;
}

/*  SOUNDBUSY  --  Return status indicating whether recording or playback
		   is currently underway. */

int soundbusy()
{
    return (recording ? 2 : 0) | (playing ? 1 : 0);
}

/*  SOUNDWRITE	--  Write a sound bite to an open file in either ASCII
		    or binary format. */

void soundwrite(isascii, fd, len, buf)
  int isascii;
  FILE *fd;
  int len;
  unsigned char *buf;
{
    if (isascii) {
        fprintf(fd, "%d\n", len);
	while (len > 0) {
	    int i = len;

	    if (i > 35)
		i = 35;
	    len -= i;
	    while (i > 0) {
                fprintf(fd, "%02X", *buf++);
		i--;
	    }
            fprintf(fd, "\n");
	}
    } else {
	long ld = len;

	fwrite(&ld, sizeof ld, 1, fd);
	fwrite(buf, len, 1, fd);
    }
}

/*  SOUNDREAD  --  Read a sound bite from a file. */

void soundread(isascii, fd, len, buf)
  int isascii;
  FILE *fd;
  int *len;
  unsigned char **buf;
{
    if (isascii) {
	long ld, id;
	unsigned char *cp;

        fscanf(fd, "%ld", &ld);

	if (ld > 0) {
	    cp = (unsigned char *) malloc(ld);
	    if (cp != NULL) {
		char broiled[82];

		*len = ld;
		*buf = cp;
		id = ld;
		while (id > 0) {
		    int ci;

                    fscanf(fd, "%02X", &ci);
		    *cp++ = ci;
		    id--;
		}
		fgets(broiled, 80, fd);
	    } else {
		*buf = NULL;
		*len = 0;
	    }
	}
    } else {
	long ld;
	unsigned char *sp;

	fread(&ld, sizeof ld, 1, fd);
	sp = (unsigned char *) malloc(ld);
	if (sp != NULL) {
	    fread(sp, ld, 1, fd);
	    *len = ld;
	    *buf = sp;
	} else {
	    *len = 0;
	    *buf = NULL;
	}
    }
}
#endif

/*  SOUNDDEST  --  Set destination for generated sound.  If "where"
		   is 0, sound goes to the built-in speaker; if
		   1, to the audio output jack. */

void sounddest(where)
  int where;
{
    AUDIO_INITINFO(&Audio_info);
    Audio_info.play.port = (where == 0 ? AUDIO_SPEAKER : AUDIO_HEADPHONE);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set output port");
    }
}

/*  SOUNDGRAB  --  Return audio information in the record queue.  */

int soundgrab(buf, len)
    char *buf;
    int len;
{
    if (recording) {
	long read_size;

	if (ioctl(audiof, FIONREAD, &read_size) < 0) {
            perror("FIONREAD ioctl failed");
	} else {
	    if (read_size > len)
		read_size = len;
	    read(audiof, buf, read_size);
	    return read_size;
	}
    }
    return 0;
}
#endif

#ifdef sgi

/*

     Silicon Graphics audio drivers adapted from the stand-alone
	     Netfone for SGI designed and implemented by:

		   Paul Schurman (schurman@frs.fi)
			    Espoo, Finland

			     16 July 1995

*/

#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <dmedia/audio.h>

#include "ulaw2linear.h"

#define TRUE  1
#define FALSE 0

#define V     (void)

/*  Local variables  */

#define BUFL	8000

static ALport audioport;	      /* Audio I/O port */
static long origParams[] = {
    AL_INPUT_RATE, 0,
    AL_OUTPUT_RATE, 0,
    AL_LEFT_SPEAKER_GAIN, 0,
    AL_RIGHT_SPEAKER_GAIN, 0,
    AL_LEFT_INPUT_ATTEN, 0,
    AL_RIGHT_INPUT_ATTEN, 0
};
static long origMute[] = {
    AL_SPEAKER_MUTE_CTL, 0
};

/*  SOUNDINIT  --  Open the sound peripheral and initialise for
		   access.  Return TRUE if successful, FALSE
		   otherwise.  */

int soundinit(iomode)
  int iomode;
{
    ALconfig audioconfig; 
    int err;
    static long params[] = {
	AL_INPUT_RATE, 8000,
	AL_OUTPUT_RATE, 8000
    };

    ALseterrorhandler(0);

    /* Get a new audioconfig and set parameters. */
  
    audioconfig = ALnewconfig();   
    ALsetsampfmt(audioconfig, AL_SAMPFMT_TWOSCOMP);
    ALsetwidth(audioconfig, AL_SAMPLE_16);
    ALsetqueuesize(audioconfig, 8000L);  
    ALsetchannels(audioconfig, AL_MONO);   

    /* Save original of all global modes we change. */ 

    ALgetparams(AL_DEFAULT_DEVICE, origMute, 2);
    ALgetparams(AL_DEFAULT_DEVICE, origParams, 12);

    /* Set input and output data rates to 8000 samples/second. */

    ALsetparams(AL_DEFAULT_DEVICE, params, 4);

    /* Open the audioport. */

    audioport = ALopenport((iomode & O_WRONLY) ? "Speaker" : "Mike",
                           (iomode & O_WRONLY) ? "w" : "r", audioconfig);
    if (audioport == (ALport) 0) {	
       err = oserror();      
       if (err == AL_BAD_NO_PORTS) {
          fprintf(stderr, " System is out of audio ports\n"); 
       } else if (err == AL_BAD_DEVICE_ACCESS) { 
          fprintf(stderr, " Couldn't access audio device\n"); 
       } else if (err == AL_BAD_OUT_OF_MEM) { 
          fprintf(stderr, " Out of memory\n"); 
       } 
       return FALSE;
    }  

    if (ALfreeconfig(audioconfig) != 0) {
       err = oserror();
       if (err == AL_BAD_CONFIG) {
          fprintf(stderr, " Config not valid");
	  return FALSE;
       }
    } 

    /* Initialized the audioport okay. */

    return TRUE;
}

/*  SOUNDTERM  --  Close the sound device.  */

void soundterm()
{
    ALsetparams(AL_DEFAULT_DEVICE, origParams, 12);
    ALsetparams(AL_DEFAULT_DEVICE, origMute, 2);
    ALcloseport(audioport);
}

/*  SOUNDRECORD  --  Begin recording of sound.	*/

int soundrecord()
{
    return TRUE;
}

/*  SOUNDPLAY  --  Begin playing a sound.  */

void soundplay(len, buf)
  int len;
  unsigned char *buf;
{
    int i;
    short abuf[BUFL];

    for (i = 0; i < len; i++) {
	abuf[i] = audio_u2s(buf[i]);
    }
    ALwritesamps(audioport, abuf, len);
}

/*  SOUNDPLAYVOL  --  Set playback volume from 0 (silence) to 100 (full on). */

void soundplayvol(value)
  int value;
{
    long par[] = {
	AL_LEFT_SPEAKER_GAIN, 0,
	AL_RIGHT_SPEAKER_GAIN, 0
    };

    par[1] = par[3] = (value * 255L) / 100;
    ALsetparams(AL_DEFAULT_DEVICE, par, 4);
}

/*  SOUNDRECGAIN  --  Set recording gain from 0 (minimum) to 100 (maximum).  */

void soundrecgain(value)
  int value;
{
    long par[] = {
	AL_LEFT_INPUT_ATTEN, 0,
	AL_RIGHT_INPUT_ATTEN, 0
    };

    par[1] = par[3] = ((100 - value) * 255L) / 100;
    ALsetparams(AL_DEFAULT_DEVICE, par, 4);
}

/*  SOUNDDEST  --  Set destination for generated sound.  If "where"
		   is 0, sound goes to the built-in speaker; if
		   1, to the audio output jack. */

void sounddest(where)
  int where;
{
    /* Since we can't mute independently, and this is used only
       for ring, always unmute the speaker and headphones. */

    static long par[] = {
	AL_SPEAKER_MUTE_CTL, AL_SPEAKER_MUTE_OFF
    };

    ALsetparams(AL_DEFAULT_DEVICE, par, 2);
}

/*  SOUNDGRAB  --  Return audio information in the record queue.  */

int soundgrab(buf, len)
    char *buf;
    int len;
{
    int i;
    short sb[BUFL];

    /* Length is hard-coded to the GSM value on SGI.  This allows
       us to use more efficient blocking sound input. */

    len = 1600;
    if (len > 0) {
       ALreadsamps(audioport, sb, len);

      
       for (i = 0; i < len; i++) {
	  buf[i] = audio_s2u(sb[i]);
       }
    }
    return len;
}
#endif
