/*

			Speak Freely for Unix
		  Network sound transmission program

	Designed and implemented in July of 1991 by John Walker

    Release 5: August 1995.  Added GSM compression, DES, IDEA, and
			     one-time pad encryption.

    Release 5.1: September 1995.  Added ADPCM compression, made
			     push-to-talk mode the default, switched
			     to smaller packets to avoid fragging the
			     datagram sender, optimised utilisation of
			     the available packet length.

    Release 5.2: September 1995.  New hat!!!!  Renamed to "Speak Freely
                             for Unix".  SGI packet length made compatible
			     with Windows version.  Platform-dependent
			     soundflush() function introduced.

*/

#include "netfone.h"

/*  Destination host descriptor.  */

struct destination {
    struct destination *dnext;
    char *server;
    struct sockaddr_in name;
    char deskey[9];		      /* Destination DES key, if any */
    char ideakey[17];		      /* Destination IDEA key, if any */
    char *otp;			      /* One-time pad */
};

static char *progname;		      /* Program name */
static int sock;		      /* Communication socket */
static struct destination *dests = NULL, *dtail;
static int compressing = FALSE;       /* Compress sound: simple 2X */
static int gsmcompress = FALSE;       /* GSM compress buffers */
static int adpcmcompress = FALSE;     /* ADPCM compress buffers */
static int squelch = 0; 	      /* Squelch level if > 0 */
#ifdef PUSH_TO_TALK
static int push = TRUE; 	      /* Push to talk mode */
static int talking = FALSE;	      /* Push to talk button state */
static int rawmode = FALSE;	      /* Is terminal in raw mode ? */
#endif
static int ring = FALSE;	      /* Force speaker & level on next pkt ? */
static int agc = FALSE; 	      /* Automatic gain control active ? */
static int rgain = 33;		      /* Current recording gain level */
static int debugging = FALSE;	      /* Debugging enabled here and there ? */
static int havesound = FALSE;         /* True if we've acquired sound input */
static char hostname[20];	      /* Host name to send with packets */
static int loopback = FALSE;	      /* Remote loopback mode */
static gsm gsmh;		      /* GSM handle */
static struct adpcm_state adpcm;      /* ADPCM compression state */
static char curkey[9] = "";           /* Current DES key if curkey[0] != 0 */
static char curideakey[17] = "";      /* Current IDEA key if curideakey[0] != 0 */
static char *curotp = NULL;	      /* One-time pad buffer */
#ifdef TINY_PACKETS
static int sound_packet;	      /* Ideal samples/packet */
#endif

#ifdef sgi
static long usticks;		      /* Microseconds per clock tick */
#endif

/*  GSMCOMP  --  Compress the contents of a sound buffer using GSM.  */

static void gsmcomp(sb)
  struct soundbuf *sb;
{
    gsm_signal src[160];
    gsm_frame dst;
    int i, j, l = 0;
    char *dp = ((char *) sb->buffer.buffer_val) + sizeof(short);

    sb->compression |= fCompGSM;
    for (i = 0; i < sb->buffer.buffer_len; i += 160) {
	for (j = 0; j < 160; j++) {
	    if ((i + j) < sb->buffer.buffer_len) {
		src[j] = audio_u2s(sb->buffer.buffer_val[i + j]);
	    } else {
		src[j] = 0;
	    }
	}
	gsm_encode(gsmh, src, dst);
	bcopy(dst, dp, sizeof dst);
	dp += sizeof dst;
	l += sizeof dst;
    }

    /* Hide original uncompressed buffer length in first 2 bytes of buffer. */

    *((short *) sb->buffer.buffer_val) = sb->buffer.buffer_len;
    sb->buffer.buffer_len = l + sizeof(short);
}

/*  ADPCMCOMP  --  Compress the contents of a sound buffer using ADPCM.  */

static void adpcmcomp(sb)
  struct soundbuf *sb;
{
    unsigned char *dp = (unsigned char *) sb->buffer.buffer_val;
    struct adpcm_state istate;

    istate = adpcm;
    sb->compression |= fCompADPCM;
    adpcm_coder_u(dp, (char *) dp, sb->buffer.buffer_len, &adpcm);
    sb->buffer.buffer_len /= 2;

    /* Hide the ADPCM encoder state at the end of this buffer.
       The shifting and anding makes the code byte-order
       insensitive. */

    dp += sb->buffer.buffer_len;
    *dp++ = ((unsigned int) istate.valprev) >> 8;
    *dp++ = istate.valprev & 0xFF;
    *dp = istate.index;
    sb->buffer.buffer_len += 3;
}

/*  ADDEST  --	Add destination host to host list.  */

static int addest(host)
  char *host;
{
    struct destination *d;
    struct hostent *hp, *gethostbyname();

    hp = gethostbyname(host);
    if (hp == 0) {
        fprintf(stderr, "%s: unknown host\n", host);
	return FALSE;
    }

    d = (struct destination *) malloc(sizeof(struct destination));
    d->dnext = NULL;
    d->server = host;
    bcopy((char *) hp->h_addr, (char *) &(d->name.sin_addr), hp->h_length);
    bcopy(curkey, d->deskey, 9);
    bcopy(curideakey, d->ideakey, 17);
    d->otp = curotp;
    d->name.sin_family = AF_INET;
    d->name.sin_port = htons(2074);
    if (dests == NULL) {
	dests = d;
    } else {
	dtail->dnext = d;
    }
    dtail = d;
    return TRUE;
}

/*  SENDPKT  --  Send a message to all active destinations.  */

static int sendpkt(sb)
  struct soundbuf *sb;
{
    struct destination *d;

    if (gsmcompress) {
	gsmcomp(sb);
    }

    if (adpcmcompress) {
	adpcmcomp(sb);
    }

    for (d = dests; d != NULL; d = d->dnext) {
	if (d->deskey[0] || d->ideakey[0] || d->otp != NULL) {
	    soundbuf ebuf;
	    int i;
	    LONG slen;

	    bcopy(sb, &ebuf, (sizeof(struct soundbuf) - BUFL) + sb->buffer.buffer_len);
	    slen = ebuf.buffer.buffer_len;

	    /* DES encryption. */

	    if (d->deskey[0]) {
		setkey(d->deskey + 1);

                /* If we're DES encrypting we must round the size of
		   the data to be sent to be a multiple of 8 so that
		   the entire DES frame is sent. */

		slen = (slen + 7) & (~7);
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with DES key.\r\n",
			    slen);
		}
		for (i = 0; i < ebuf.buffer.buffer_len; i += 8) {

		    /* Apply cipher block chaining within the packet. */

		    if (i > 0) {
			int j;

			for (j = 0; j < 8; j++) {
			    ebuf.buffer.buffer_val[(i + j)] ^=
				ebuf.buffer.buffer_val[(i + j) - 8];
			}
		    }
		    endes(ebuf.buffer.buffer_val + i);
		}
		ebuf.compression |= fEncDES;
	    }

	    /* IDEA encryption. */

	    if (d->ideakey[0]) {
		unsigned short iv[4];

		bzero(iv, sizeof(iv));
		initcfb_idea(iv, d->ideakey + 1, FALSE);

                /* If we're IDEA encrypting we must round the size of
		   the data to be sent to be a multiple of 8 so that
		   the entire IDEA frame is sent. */

		slen = (slen + 7) & (~7);
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with IDEA key.\r\n",
			    slen);
		}
		ideacfb(ebuf.buffer.buffer_val, slen);
		close_idea();
		ebuf.compression |= fEncIDEA;
	    }

	    /* One-time pad encryption. */

	    if (d->otp != NULL) {
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with one-time pad.\r\n",
			    ebuf.buffer.buffer_len);
		}
		for (i = 0; i < ebuf.buffer.buffer_len; i ++) {
		    ebuf.buffer.buffer_val[i] ^= d->otp[i];
		}
		ebuf.compression |= fEncOTP;
	    }
	    if (sendto(sock, &ebuf, (sizeof(struct soundbuf) - BUFL) + slen,
		0, (struct sockaddr *) &(d->name), sizeof d->name) < 0) {
                perror("sending datagram message");
		return FALSE;
	    }
	} else {
	    if (sendto(sock, sb, (sizeof(struct soundbuf) - BUFL) +
				    sb->buffer.buffer_len,
		0, (struct sockaddr *) &(d->name), sizeof d->name) < 0) {
                perror("sending datagram message");
		return FALSE;
	    }
	}
    }
    return TRUE;
}

/*  EXITING  --  Catch as many program termination signals as
		 possible and restore initial audio modes before
		 we exit.  */

static void exiting()
{
#ifdef PUSH_TO_TALK
    if (rawmode) {
        fprintf(stderr, "\r      \r");
	fcntl(fileno(stdin), F_SETFL, 0);
	noraw();
	echo();
	endwin();
    }
#endif
    if (havesound) {
	if (debugging) {
            fprintf(stderr, "Restoring audio modes at exit.\n");
	}
	soundterm();
	exit(0);
    }
}

/*  TERMCHAR  --  Check for special characters from console when
		  in raw mode.	*/

static void termchar(ch)    
  int ch;
{
#define Ctrl(x) ((x) - '@')
    if (ch == 'q' || ch == 'Q' || ch == 27 ||
        ch == Ctrl('C') || ch == Ctrl('D')) {
	exiting();
    }
}

/*  SENDFILE  --  Send a file or, if the file name is NULL or a
		  single period, send real-time sound input. */

static int sendfile(f)
  char *f;
{ 
    soundbuf netbuf;
#define buf netbuf.buffer.buffer_val
    int bread, nread;
    FILE *afile = NULL;

#ifdef TINY_PACKETS

    /* Compute the number of sound samples needed to fill a
       packet of TINY_PACKETS bytes. */

    sound_packet = ((TINY_PACKETS - ((sizeof(soundbuf) - BUFL))) *
		    (compressing ? 2 : 1));
    if (gsmcompress) {
	sound_packet = compressing ? 3200 : 1600;
    }
    if (adpcmcompress) {
	sound_packet *= 2;
	sound_packet -= 4;	      /* Leave room for state at the end */
    }
#ifdef SHOW_PACKET_SIZE
    printf("Samples per packet = %d\n", sound_packet);
#endif
#endif

    strcpy(netbuf.sendinghost, hostname);
    if (f != NULL && (strcmp(f, ".") != 0)) {
	char magic[4];

        afile = fopen(f, "r");
	if (afile == NULL) {
            fprintf(stderr, "Unable to open sound file %s.\n", f);
	    return 2;
	}

	/* If the file has a Sun .au file header, skip it.
	   Note that we still blithely assume the file is
	   8-bit ISDN u-law encoded at 8000 samples per
	   second. */

	fread(magic, 1, sizeof(long), afile);
        if (bcmp(magic, ".snd", 4) == 0) {
	    long startpos;

	    fread(&startpos, sizeof(long), 1, afile);
#ifdef _M_I86
	    revlong(&startpos);
#endif
	    fseek(afile, startpos, 0);
	} else {
	    fseek(afile, 0L, 0);
	}
    }

    /* Send a file */

    if (afile) {
#ifdef sun
	struct timeb t1, t2;
#endif
	long et;

	while (
#ifdef sun
		    ftime(&t1),
#endif
		    (bread = nread =
		    fread(buf, 1,
#ifdef TINY_PACKETS
				  sound_packet,
#else
				  (gsmcompress ? 1600 : BUFL),
#endif
				  afile)) > 0) {
	    netbuf.compression = FALSE | (ring ? (fSetDest | fDestSpkr) : 0);
	    ring = FALSE;
	    netbuf.compression |= debugging ? fDebug : 0;
	    netbuf.compression |= loopback ? fLoopBack : 0;
	    if (compressing) {
		int i;

		nread /= 2;
		for (i = 1; i < nread; i++) {
		    buf[i] = buf[i * 2];
		}
		netbuf.compression |= fComp2X;
	    }
	    netbuf.buffer.buffer_len = nread;
	    if (!sendpkt(&netbuf)) {
		fclose(afile);
		return 1;
	    }

            /* The following code is needed because when we're reading
	       sound from a file, as opposed to receiving it in real
	       time from the CODEC, we must meter out the samples
	       at the rate they will actually be played by the destination
	       machine.  For a 8000 samples per second, this amounts
	       to 125 microseconds per sample, minus the time we spent
	       compressing the data (which is substantial for GSM) and
	       a fudge factor, kOverhead, which accounts for the time
	       spent in executing the delay itself and getting control
               back after it's done.  If sound files pause periodically
               (when the sending machine isn't loaded), you may need
               to reduce the delay parameters.  If they're too low,
	       however, data will be lost when sending long sound files. */

#ifdef TINY_PACKETS
#define kOverhead   8000
#else
#define kOverhead   3000
#endif

#ifdef sun
	    ftime(&t2);
	    et = ((bread * 125) -
		  1000 * (((t2.time - t1.time) * 1000) +
			   (t2.millitm - t1.millitm))) - kOverhead;
#endif

#ifdef sgi
	    et = (bread * 125) - kOverhead;
#endif

	    if (et > 0) {
#ifdef sun
		usleep(et);
#endif
#ifdef sgi
		sginap(et / usticks);
#endif
	    }
	    if (debugging) {
                fprintf(stderr, "Sent %d samples from %s in %d bytes.\r\n",
			bread, f, netbuf.buffer.buffer_len);
	    }
	}

	if (debugging) {
            fprintf(stderr, "Sent sound file %s.\r\n", f);
	}
	fclose(afile);
    } else {

#ifdef PUSH_TO_TALK
	if (push) {
	    char c;

	    fprintf(stderr,
                "Space bar switches talk/pause, Esc or \"q\" to quit\nPause: ");
	    fflush(stderr);
	    initscr();
	    noecho();
	    raw();
	    rawmode = TRUE;
	    read(fileno(stdin), &c, 1);
	    termchar(c);
            fprintf(stderr, "\rTalk:  ");
	    fcntl(fileno(stdin), F_SETFL, O_NDELAY);
	    talking = TRUE;
	}
#endif

	/* Send real-time sound. */

	if (!soundinit(O_RDONLY /* | O_NDELAY */ )) {
            fprintf(stderr, "Unable to initialise audio.\n");
	    return 2;
	}
	havesound = TRUE;
	signal(SIGHUP, exiting);     /* Set signal to handle termination */
	signal(SIGINT, exiting);     /* Set signal to handle termination */
	signal(SIGTERM, exiting);    /* Set signal to handle termination */
	if (agc) {
	    soundrecgain(rgain);      /* Set initial record level */
	}
	if (soundrecord()) {
	    while (TRUE) {
		int soundel = soundgrab(buf,
#ifdef TINY_PACKETS
					     sound_packet
#else
					     sizeof buf
#endif
				       );

		unsigned char *bs = (unsigned char *) buf;

#ifdef PUSH_TO_TALK
		if (push) {
		    char c;

		    if (read(fileno(stdin), &c, 1) > 0) {
			termchar(c);
			talking = !talking;
                        fprintf(stderr, "\r%s:  ", talking ? "Talk" : "Pause");
			fflush(stderr);
			fcntl(fileno(stdin), F_SETFL, talking ? O_NDELAY : 0);
			if (talking) {
			    /* Discard all backlog sound input. */
			    soundflush();
			}
		    }
/*
		    if (!talking) {
			continue;
		    }
*/
		}
#endif

		if (soundel > 0) {
		    register unsigned char *start = bs;
		    register int j;
		    int squelched = (squelch > 0), osl = soundel;

		    /* If entire buffer is less than squelch, ditch it. */

		    if (squelch > 0) {
			for (j = 0; j < soundel; j++) {
			    if (((*start++ & 0x7F) ^ 0x7F) > squelch) {
				squelched = FALSE;
				break;
			    }
			}
		    }

		    if (squelched) {
			if (debugging) {
                            printf("Entire buffer squelched.\n");
			}
		    } else {
			netbuf.compression = FALSE | (ring ? (fSetDest | fDestSpkr) : 0);
			netbuf.compression |= debugging ? fDebug : 0;
			netbuf.compression |= loopback ? fLoopBack : 0;

			/* If automatic gain control is enabled,
			   ride the gain pot to fill the dynamic range
			   optimally. */

			if (agc) {
			    register unsigned char *start = bs;
			    register int j;
			    long msamp = 0;

			    for (j = 0; j < soundel; j++) {
				int s = audio_u2s(*start++);

				msamp += (s < 0) ? -s : s;
			    }
			    msamp /= soundel;
			    if (msamp < 6000) {
				if (rgain < 100) {
				    soundrecgain(++rgain);
				}
			    } else if (msamp > 7000) {
				if (rgain > 1) {
				    soundrecgain(--rgain);
				}
			    }
			}

			ring = FALSE;
			if (compressing) {
			    int i;

			    soundel /= 2;
			    for (i = 1; i < soundel; i++) {
				buf[i] = buf[i * 2];
			    }
			    netbuf.compression |= fComp2X;
			}
			netbuf.buffer.buffer_len = soundel;
			if (!sendpkt(&netbuf)) {
			    return 1;
			}
			if (debugging) {
                            fprintf(stderr, "Sent %d audio samples in %d bytes.\r\n",
				    osl, netbuf.buffer.buffer_len);
			}
		    }
		} else {
#ifdef sun
		    usleep(100000L);  /* Wait for some sound to arrive */
#endif
		}
	    }
	} else {
            fprintf(stderr, "Unable to start recording.\n");
	    return 2;
	}
    }
    return 0;
}

/*  USAGE  --  Print how-to-call information.  */

static void usage()
{
    V fprintf(stderr, "%s  --  Speak Freely sound sender.\n", progname);
    V fprintf(stderr, "            Release 5.2, September 1995.\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "Usage: %s hostname [options] [ file1 / . ]...\n", progname);
    V fprintf(stderr, "Options: (* indicates defaults)\n");
#ifdef PUSH_TO_TALK
    V fprintf(stderr, "           -A         Always transmit unless squelched\n");
    V fprintf(stderr, "     *     -B         Push to talk using keyboard\n");
#endif
    V fprintf(stderr, "           -C         Compress subsequent sound\n");
    V fprintf(stderr, "           -D         Enable debug output\n");
    V fprintf(stderr, "           -E[key]    Emit session key string\n");
    V fprintf(stderr, "           -F         ADPCM compression\n");
    V fprintf(stderr, "           -G         Automatic gain control\n");
    V fprintf(stderr, "           -Ikey      IDEA encrypt with key\n");
    V fprintf(stderr, "           -Kkey      DES encrypt with key\n");
    V fprintf(stderr, "           -L         Remote loopback\n");
    V fprintf(stderr, "     *     -M         Manual record gain control\n");
    V fprintf(stderr, "     *     -N         Do not compress subsequent sound\n");
    V fprintf(stderr, "           -Ofile     Use file as one-time pad\n");
    V fprintf(stderr, "           -Phostname Party line, add host to list\n");
    V fprintf(stderr, "     *     -Q         Disable debug output\n");
    V fprintf(stderr, "           -R         Ring--force volume, output to speaker\n");
    V fprintf(stderr, "           -Sn        Squelch at level n (0-255)\n");
    V fprintf(stderr, "           -T         Telephone (GSM) compression\n");
    V fprintf(stderr, "           -U         Print this message\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "by John Walker\n");
    V fprintf(stderr, "   E-mail: kelvin@fourmilab.ch\n");
    V fprintf(stderr, "   WWW:    http://www.fourmilab.ch/\n");
}

/*  Main program.  */

main(argc, argv)
  int argc;
  char *argv[];
{
    int i, j, k, l, sentfile = 0;
    FILE *fp;
    struct MD5Context md5c;
    char md5key[16], md5key1[16];
    char s[1024];

    progname = argv[0];
    gethostname(hostname, sizeof hostname);
    gsmh = gsm_create();

#ifdef sgi
    usticks = 1000000 / CLK_TCK;
#endif

    /* Create the socket used to send data. */

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0) {
        perror("opening datagram socket");
	return 1;
    }

    /*	Process command line options.  */

    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

#ifdef PUSH_TO_TALK
                case 'A':             /* -A  --  Always transmit (no push to talk) */
		    push = FALSE;
		    break;

                case 'B':             /* -B  -- Push to talk (button) */
		    if (isatty(fileno(stdin))) {
			push = TRUE;
		    }
		    break;
#endif

                case 'C':             /* -C  -- Compress sound samples */
		    compressing = TRUE;
		    break;

                case 'D':             /* -D  --  Enable debug output  */
		    debugging = TRUE;
		    break;

                case 'E':             /* -E  --  Emit session key and exit */
		    strcpy(s, op + 1);
                    sprintf(s + strlen(s), "%u", getpid());
                    sprintf(s + strlen(s), "%u", getppid());
		    V getcwd(s + strlen(s), 256);
                    sprintf(s + strlen(s), "%u", clock());
		    V cuserid(s + strlen(s));
                    sprintf(s + strlen(s), "%u", time(NULL));
                    sprintf(s + strlen(s), "%u", gethostid());
		    getdomainname(s + strlen(s), 256);
		    gethostname(s + strlen(s), 256);
                    sprintf(s + strlen(s), "%u", getuid());
                    sprintf(s + strlen(s), "%u", getgid());
		    MD5Init(&md5c);
		    MD5Update(&md5c, s, strlen(s));
		    MD5Final(md5key, &md5c);
                    sprintf(s + strlen(s), "%u", (time(NULL) + 65121) ^ 0x375F);
		    MD5Init(&md5c);
		    MD5Update(&md5c, s, strlen(s));
		    MD5Final(md5key1, &md5c);
		    init_idearand(md5key, md5key1, time(NULL));
		    for (j = k = 0; j < 16; j++) {
			unsigned char rb = idearand();

#define Rad32(x) ((x) + 'A')
			s[k++] = Rad32((rb >> 4) & 0xF);
			s[k++] = Rad32(rb & 0xF);
			if (j & 1) {
                            s[k++] = '-';
			}
		    }
		    close_idearand();
		    s[--k] = 0;
                    printf("%s\n", s);
		    return 0;

                case 'F':             /* -F -- ADPCM compression */
		    if (gsmcompress) {
                        fprintf(stderr, "Cannot use GSM (-t) and ADPCM (-f) compression simultaneously.\n");
			return 2;
		    }
		    adpcmcompress = TRUE;
		    break;

                case 'G':             /* -G  --  Automatic gain control */
		    agc = TRUE;
		    break;

                case 'I':             /* -Ikey  --  Set IDEA key */
		    if (strlen(op + 1) == 0) {
			curideakey[0] = FALSE;
		    } else {
			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(curideakey + 1, &md5c);
			curideakey[0] = TRUE;
			if (debugging) {
                            fprintf(stderr, "IDEA key:");
			    for (j = 1; j < 17; j++) {
                                fprintf(stderr, " %02X", (curideakey[j] & 0xFF));
			    }
                            fprintf(stderr, "\n");
			}
		    }
		    break;

                case 'K':             /* -Kkey  --  Set DES key */
		    desinit(1);       /* Initialise the DES library */
		    if (strlen(op + 1) == 0) {
			curkey[0] = FALSE;
		    } else {
			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(md5key, &md5c);
			for (j = 0; j < 8; j++) {
			    curkey[j + 1] = (char)
					  ((md5key[j] ^ md5key[j + 8]) & 0x7F);
			}
			if (debugging) {
                            fprintf(stderr, "DES key:");
			    for (j = 0; j < 8; j++) {
                                fprintf(stderr, " %02X", (curkey[j + 1] & 0xFF));
			    }
                            fprintf(stderr, "\n");
			}
			curkey[0] = TRUE;
		    }
		    break;

                case 'L':             /* -L  --  Remote loopback */
		    loopback = TRUE;
		    break;

                case 'M':             /* -M  --  Manual record gain control */
		    agc = FALSE;
		    break;

                case 'N':             /* -N  --  Do not compress sound samples */
		    compressing = FALSE;
		    gsmcompress = FALSE;
		    adpcmcompress = FALSE;
		    break;

                case 'O':             /* -Ofile -- Use file as one-time pad */
		    if (op[1] == 0) {
			curotp = NULL; /* Switch off one-time pad */
		    } else {
                        fp = fopen(op + 1, "r");
			if (fp == NULL) {
                            perror("Cannot open one-time pad file");
			    return 2;
			}
			curotp = malloc(BUFL);
			if (curotp == NULL) {
                            fprintf(stderr, "Cannot allocate one-time pad buffer.\n");
			    return 2;
			}
			l = fread(curotp, 1, BUFL, fp);
			if (l == 0) {
                            /* Idiot supplied void one-time pad.  Give 'im
			       what he asked for: no encryption. */
			    curotp[0] = 0;
			    l = 1;
			}
			fclose(fp);
			/* If the file is shorter than the maximum buffer
			   we may need to encrypt, replicate the key until
			   the buffer is filled. */
			j = l;
			k = 0;
			while (j < BUFL) {
			    curotp[j++] = curotp[k++];
			    if (k >= l) {
				k = 0;
			    }
			}
		    }
		    break;

                case 'P':             /* -Phost  --  Copy output to host  */
		    if (!addest(op + 1)) {
			return 1;
		    }
		    break;

                case 'Q':             /* -Q  --  Disable debug output  */
		    debugging = FALSE;
		    break;

                case 'R':             /* -R  --  Ring: divert output to speaker */
		    ring = TRUE;
		    break;

                case 'S':             /* -Sn  --  Squelch at level n */
		    if (strlen(op + 1) == 0) {
			squelch = 50; /* Default squelch */
		    } else {
			squelch = atoi(op + 1);
		    }
		    break;

                case 'T':             /* -T -- Telephone (GSM) compression */
		    if (adpcmcompress) {
                        fprintf(stderr, "Cannot use GSM (-t) and ADPCM (-f) compression simultaneously.\n");
			return 2;
		    }
		    gsmcompress = TRUE;
		    break;

                case 'U':             /* -U  --  Print usage information */
                case '?':             /* -?  --  Print usage information */
		    usage();
		    return 0;
	    }
	} else {
	    if (dests == NULL) {
		if (!addest(op)) {
		    return 1;
		}
	    } else {
		int ok = sendfile(op);
		if (ok != 0)
		    return ok;
		sentfile++;
	    }
	}
    }

    if (dests == NULL) {
	usage();
    } else {
	if (sentfile == 0) {
	    return sendfile(NULL);
	}
    }

    gsm_destroy(gsmh);
    desdone();
    return 0;
}
