/*

	Sound interface for Netfone

	Designed and implemented in July of 1990 by John Walker

*/

#define BUFL	8000

#include "netfone.h"

#ifdef BSD_like

#include <sys/dir.h>
#include <sys/file.h>

#include <math.h>
#include <errno.h>

#include <sys/ioctl.h>
#ifdef sun
#include <sun/audioio.h>
#else /* FreeBSD / Linux */
#include <machine/soundcard.h>
#define AUDIO_MIN_GAIN 0
#define AUDIO_MAX_GAIN 255
static int abuf_size;
#endif

#define SoundFile       "/dev/audio"
#ifdef sun
#define AUDIO_CTLDEV    "/dev/audioctl"
#else
#define AUDIO_CTLDEV    "/dev/mixer"
#endif

#define MAX_GAIN	100

struct sound_buf {
    struct sound_buf *snext;	      /* Next sound buffer */
    int sblen;			      /* Length of this sound buffer */
    unsigned char sbtext[2];	      /* Actual sampled sound */
};

/*  Local variables  */

static int audiof = -1; 	      /* Audio device file descriptor */
static int Audio_fd;		      /* Audio control port */
#ifdef sun
static audio_info_t Audio_info;       /* Current configuration info */
#endif
struct sound_buf *sbchain = NULL,     /* Sound buffer chain links */
		 *sbtail = NULL;
static int sbtotal = 0; 	      /* Total sample bytes in memory */
static int playing = FALSE;	      /* Replay in progress ? */
/* static int playqsize;  */		    /* Output queue size */
static int playlen = 0; 	      /* Length left to play */
static unsigned char *playbuf= NULL;  /* Current play pointer */
static int squelch = 0; 	      /* Squelch value */

/* Convert local gain into device parameters */

static unsigned scale_gain(g)
  unsigned g;
{
    return (AUDIO_MIN_GAIN + (unsigned)
	((int) ((((double) (AUDIO_MAX_GAIN - AUDIO_MIN_GAIN)) *
	((double) g / (double) MAX_GAIN)) + 0.5)));
}


/*  SOUNDINIT  --  Open the sound peripheral and initialise for
		   access.  Return TRUE if successful, FALSE
		   otherwise.  */

int soundinit(iomode)
  int iomode;
{
    assert(audiof == -1);
    if ((audiof = open(SoundFile, iomode)) >= 0) {

	if ((Audio_fd = open(AUDIO_CTLDEV, O_RDWR)) < 0) {
	    perror(AUDIO_CTLDEV);
	    return FALSE;
	}
/*fcntl(audiof, F_SETFL, O_NDELAY);*/
#ifndef sun
	if (ioctl(audiof, SNDCTL_DSP_NONBLOCK, NULL) < 0) {
            perror("SNDCTL_DSP_NONBLOCK");
	    return FALSE;
	}
	if (ioctl(audiof, SNDCTL_DSP_GETBLKSIZE, &abuf_size) < 0) {
            perror("SNDCTL_DSP_GETBLKSIZE");
	    return FALSE;
	}
#else
#endif
	return TRUE;
    }
    return FALSE;
}

/*  SOUNDTERM  --  Close the sound device.  */

void soundterm()
{
	if (audiof >= 0) {
	   V close(audiof);
	   V close(Audio_fd);
	   audiof = -1;
	}
}

/*  SOUNDPLAY  --  Begin playing a sound.  */

void soundplay(len, buf)
  int len;
  unsigned char *buf;
{
    int ios;

    assert(audiof != -1);
    while (TRUE) {
	ios = write(audiof, buf, len);
	if (ios == -1) {
	    usleep(100000);
	} else {
	    if (ios < len) {
		buf += ios;
		len -= ios;
	    } else {
		break;
	    }
	}
    }
}

/*  SOUNDPLAYVOL  --  Set playback volume from 0 (silence) to 100 (full on). */

void soundplayvol(value)
  int value;
{
#ifdef sun
    AUDIO_INITINFO(&Audio_info);
    Audio_info.play.gain = scale_gain(value);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set play volume");
    }
#else
   int arg;

   value = (value * 100 / 255) & 0x7f;
   arg	 = (value << 8) | value;

   if (ioctl(Audio_fd, MIXER_WRITE(SOUND_MIXER_PCM), &arg) < 0)
        perror("SOUND_MIXER_PCM");
#endif
}

/*  SOUNDRECGAIN  --  Set recording gain from 0 (minimum) to 100 (maximum).  */

void soundrecgain(value)
  int value;
{
#ifdef sun
    AUDIO_INITINFO(&Audio_info);
    Audio_info.record.gain = scale_gain(value);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set record gain");
    }
#else
    int arg, i1, i2, i3;

    value = (value * 100 / 255) & 0x7f;
    arg   = (value << 8) | value;

    i1 = ioctl(Audio_fd, SOUND_MIXER_WRITE_LINE, &arg);
    i2 = ioctl(Audio_fd, SOUND_MIXER_WRITE_MIC,  &arg);
    i3 = ioctl(Audio_fd, SOUND_MIXER_WRITE_CD,	 &arg);
    if (i1 < 0 && i2 < 0 && i3 < 0)
        perror("SOUND_MIXER_WRITE_*");
#endif
}

/*  SOUNDDEST  --  Set destination for generated sound.  If "where"
		   is 0, sound goes to the built-in speaker; if
		   1, to the audio output jack. */

void sounddest(where)
  int where;
{
#ifdef sun
    AUDIO_INITINFO(&Audio_info);
    Audio_info.play.port = (where == 0 ? AUDIO_SPEAKER : AUDIO_HEADPHONE);
    if (ioctl(Audio_fd, AUDIO_SETINFO, &Audio_info) < 0) {
        perror("Set output port");
    }
#endif
}

/*  SOUNDGRAB  --  Return audio information in the record queue.  */

int soundgrab(buf, len)
    char *buf;
    int len;
{
    long read_size;
    int c;

#ifdef sun
#ifndef BlockingIO
    if (ioctl(audiof, FIONREAD, &read_size) < 0) {
        perror("soundgrab FIONREAD ioctl failed");
    } else {
	if (read_size > len) {
	    read_size = len;
	}
#else
    read_size = len;
#endif
#else
	read_size = len;
	if (read_size > abuf_size) {
	    read_size = abuf_size;
	}
#endif
	c = read(audiof, buf, read_size);
	if (c < 0 && errno == EAGAIN) {
	    c = 0;
	}
	if (c < 0) {
            perror("soundgrab");
	}
	return c;
#ifndef BlockingIO
    }
    return 0;
#endif
}

/*  SOUNDFLUSH	--  Flush any queued sound.  */

void soundflush()
{
    char sb[BUFL];
#ifndef sun
    int c;
#endif

    while (TRUE) {
#ifndef sun
	c = read(audiof, sb, BUFL < abuf_size ? BUFL : abuf_size);
	if (c < 0 && errno == EAGAIN)
	    c = 0;
	if (c < 0)
            perror("soundflush");
	if (c <= 0)
	    break;
#else
	long read_size;

#ifndef BlockingIO
	if (ioctl(audiof, FIONREAD, &read_size) < 0) {
            perror("soundflush FIONREAD ioctl failed");
	    break;
#else
	if (read_size = 400, TRUE) {
#endif
	} else if (read_size > 400) {
	    if (read_size > BUFL) {
		read_size = BUFL;
	    }
	    read(audiof, sb, read_size);
#ifdef BlockingIO
	    break;
#endif
	} else {
	    break;
	}
#endif
    }
}
#endif

#ifdef sgi

/*

     Silicon Graphics audio drivers adapted from the stand-alone
	     Netfone for SGI designed and implemented by:

		   Paul Schurman (schurman@frs.fi)
			    Espoo, Finland

			     16 July 1995

*/

#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <dmedia/audio.h>

#include "ulaw2linear.h"

/*  Local variables  */

static ALport audioport;	      /* Audio I/O port */
static long origParams[] = {
    AL_INPUT_RATE, 0,
    AL_OUTPUT_RATE, 0,
    AL_LEFT_SPEAKER_GAIN, 0,
    AL_RIGHT_SPEAKER_GAIN, 0,
    AL_LEFT_INPUT_ATTEN, 0,
    AL_RIGHT_INPUT_ATTEN, 0
};
static long origMute[] = {
    AL_SPEAKER_MUTE_CTL, 0
};

/*  SOUNDINIT  --  Open the sound peripheral and initialise for
		   access.  Return TRUE if successful, FALSE
		   otherwise.  */

int soundinit(iomode)
  int iomode;
{
    ALconfig audioconfig; 
    int err;
    static long params[] = {
	AL_INPUT_RATE, 8000,
	AL_OUTPUT_RATE, 8000
    };

    ALseterrorhandler(0);

    /* Get a new audioconfig and set parameters. */
  
    audioconfig = ALnewconfig();   
    ALsetsampfmt(audioconfig, AL_SAMPFMT_TWOSCOMP);
    ALsetwidth(audioconfig, AL_SAMPLE_16);
    ALsetqueuesize(audioconfig, 8000L);  
    ALsetchannels(audioconfig, AL_MONO);   

    /* Save original of all global modes we change. */ 

    ALgetparams(AL_DEFAULT_DEVICE, origMute, 2);
    ALgetparams(AL_DEFAULT_DEVICE, origParams, 12);

    /* Set input and output data rates to 8000 samples/second. */

    ALsetparams(AL_DEFAULT_DEVICE, params, 4);

    /* Open the audioport. */

    audioport = ALopenport((iomode & O_WRONLY) ? "Speaker" : "Mike",
                           (iomode & O_WRONLY) ? "w" : "r", audioconfig);
    if (audioport == (ALport) 0) {	
       err = oserror();      
       if (err == AL_BAD_NO_PORTS) {
          fprintf(stderr, " System is out of audio ports\n"); 
       } else if (err == AL_BAD_DEVICE_ACCESS) { 
          fprintf(stderr, " Couldn't access audio device\n"); 
       } else if (err == AL_BAD_OUT_OF_MEM) { 
          fprintf(stderr, " Out of memory\n"); 
       } 
       return FALSE;
    }  

    if (ALfreeconfig(audioconfig) != 0) {
       err = oserror();
       if (err == AL_BAD_CONFIG) {
          fprintf(stderr, " Config not valid");
	  return FALSE;
       }
    } 

    /* Initialized the audioport okay. */

    return TRUE;
}

/*  SOUNDTERM  --  Close the sound device.  */

void soundterm()
{
    ALsetparams(AL_DEFAULT_DEVICE, origParams, 12);
    ALsetparams(AL_DEFAULT_DEVICE, origMute, 2);
    ALcloseport(audioport);
}

/*  SOUNDPLAY  --  Begin playing a sound.  */

void soundplay(len, buf)
  int len;
  unsigned char *buf;
{
    int i;
    short abuf[BUFL];

    for (i = 0; i < len; i++) {
	abuf[i] = audio_u2s(buf[i]);
    }
    ALwritesamps(audioport, abuf, len);
}

/*  SOUNDPLAYVOL  --  Set playback volume from 0 (silence) to 100 (full on). */

void soundplayvol(value)
  int value;
{
    long par[] = {
	AL_LEFT_SPEAKER_GAIN, 0,
	AL_RIGHT_SPEAKER_GAIN, 0
    };

    par[1] = par[3] = (value * 255L) / 100;
    ALsetparams(AL_DEFAULT_DEVICE, par, 4);
}

/*  SOUNDRECGAIN  --  Set recording gain from 0 (minimum) to 100 (maximum).  */

void soundrecgain(value)
  int value;
{
    long par[] = {
	AL_LEFT_INPUT_ATTEN, 0,
	AL_RIGHT_INPUT_ATTEN, 0
    };

    par[1] = par[3] = ((100 - value) * 255L) / 100;
    ALsetparams(AL_DEFAULT_DEVICE, par, 4);
}

/*  SOUNDDEST  --  Set destination for generated sound.  If "where"
		   is 0, sound goes to the built-in speaker; if
		   1, to the audio output jack. */

void sounddest(where)
  int where;
{
    /* Since we can't mute independently, and this is used only
       for ring, always unmute the speaker and headphones. */

    static long par[] = {
	AL_SPEAKER_MUTE_CTL, AL_SPEAKER_MUTE_OFF
    };

    ALsetparams(AL_DEFAULT_DEVICE, par, 2);
}

/*  SOUNDGRAB  --  Return audio information in the record queue.  */

int soundgrab(buf, len)
    char *buf;
    int len;
{
    int i;
    short sb[BUFL];
/*  long filled = ALgetfilled(audioport);

    if (len > filled) {
	len = filled;
    }
len=488;
*/
/*len=1600;*/
    if (len > 0) {
       ALreadsamps(audioport, sb, len);

      
       for (i = 0; i < len; i++) {
	  buf[i] = audio_s2u(sb[i]);
       }
    }
    return len;
}

/*  SOUNDFLUSH	--  Flush any queued sound.  */

void soundflush()
{
    short sb[BUFL];

    while (TRUE) {
	long l = ALgetfilled(audioport);

	if (l < 400) {
	    break;
	} else {
	    if (l > BUFL) {
		l = BUFL;
	    }
	    ALreadsamps(audioport, sb, l);
	}
    }
}
#endif
