/*

			Speak Freely for Unix
			 Sound Output Server

	Designed and implemented in July of 1991 by John Walker.

    Release 5: August 1995.  Added GSM compression, DES, IDEA, and
			     key file encryption.

    Release 5.1: September 1995.  Added ADPCM compression, connection
			     logging (-v) and overrides for debug (-d)
			     and quiet (-q) modes.

    Release 5.2: September 1995.  Added -rfile option to make an .au
			     file copy of all sound received in the named
			     file.  Provides a crude answering machine
			     facility.

*/

#include "netfone.h"

static int audiok = FALSE;	      /* Audio initialised flag */
static int audiotime = 0;	      /* Audio timeout counter */
static int debugging = FALSE;	      /* Debugging enabled */
static int debugforce = 0;	      /* Debugging forced / prevented ? */
static struct adpcm_state adpcm;      /* ADPCM compression state */
static gsm gsmh;		      /* GSM handle */
static char deskey[9] = "";           /* DES key, if any */
static char ideakey[17] = "";         /* IDEA key, if any */
static char *curotp = NULL;	      /* Key file, if any */
struct sockaddr_in from;	      /* Sending host address */
static int fromlen;		      /* Length of sending host address */
static int showhosts = FALSE;	      /* Show host names that connect */
static int hosttimeout = 60;	      /* Consider host idle after this time */
static char *prog;		      /* Program name */ 
static FILE *record = NULL;	      /* Answering machine record file */
static char *pgppass = NULL;	      /* PGP secret key password */
#ifdef HALF_DUPLEX
static struct in_addr localhost;      /* Our internet address */
static int halfDuplexMuted = FALSE;   /* Muted by half-duplex transmission */
#endif

struct connection {
    struct connection *con_next;      /* Next connection */
    struct in_addr con_addr;	      /* Host Internet address */
    char pgpkey[17];		      /* PGP key for connection */
    char keymd5[16];		      /* Digest of key file */
    int con_timeout;		      /* Connection timeout */
    char con_hostname[264];	      /* Host name */
};

struct connection *conn = NULL;       /* Chain of current connections */

#define Debug	((debugforce != 0) ? (debugforce > 0) : debugging)

#define TickTock    10		      /* Alarm interval in seconds */

/*  GSMDECOMP  --  Uncompress the contents of a sound buffer using GSM.  */

static void gsmdecomp(sb)
  struct soundbuf *sb;
{
    gsm_signal dst[160];
    int i, j, l = 0;
    char *dpx = ((char *) sb->buffer.buffer_val) + sizeof(short);
    static char dcb[BUFL];
    short declen = *((short *) sb->buffer.buffer_val);

    if (declen <= 0 || declen > 1600) {
	declen = 1600;
    }
    for (i = 0; i < sb->buffer.buffer_len - sizeof(short);
		i += sizeof(gsm_frame)) {
	gsm_decode(gsmh, (gsm_byte *) dpx, dst);
	dpx += sizeof(gsm_frame);
	for (j = 0; j < 160; j++) {
	    dcb[l++] = audio_s2u(dst[j]);
	}
    }
    bcopy(dcb, sb->buffer.buffer_val, declen);
    sb->buffer.buffer_len = declen;
}

/*  ADPCMDECOMP  --  Decompress the contents of a sound buffer using ADPCM.  */

static void adpcmdecomp(sb)
  struct soundbuf *sb;
{
    char *dp = (char *) sb->buffer.buffer_val;
    unsigned char *sp;
    unsigned char dob[TINY_PACKETS * 2];

    /* Restore the decoder state from the copy saved in the packet,
       independent of the byte order of the machine we're running on. */

    sp = (unsigned char *) dp + (sb->buffer.buffer_len - 3);
    adpcm.valprev = (sp[0] << 8) | sp[1];
    adpcm.index = sp[2];
    sb->buffer.buffer_len -= 3;

    adpcm_decoder_u(dp, dob, sb->buffer.buffer_len * 2, &adpcm);
    sb->buffer.buffer_len *= 2;
    bcopy(dob, dp, sb->buffer.buffer_len);
}

/*  ETIME  --  Edit time and date for log messages.  */

static char *etime()
{
    struct tm *t;
    time_t clock;
    static char s[20];

    time(&clock);
    t = localtime(&clock);
    sprintf(s, "%02d-%02d %02d:%02d", t->tm_mon + 1, t->tm_mday,
	       t->tm_hour, t->tm_min);
    return s;
}

/*  RELEASE  --  Signal-catching function which releases the audio
                 device if we haven't received anything to play in
		 the last minute. */

static void release()
{
    struct connection *c, *l, *n;
    int actives = 0;

    /* Mark idle any hosts that haven't sent us anything recently. */

    c = conn;
    l = NULL;
    while (c != NULL) {
	if (c->con_timeout < hosttimeout) {
	    c->con_timeout += TickTock;
	    actives++;
	}
	n = c->con_next;
	if (c->con_timeout == hosttimeout) {
	    actives--;		      /* Oops--just went inactive */
	    c->con_timeout++;
            fprintf(stderr, "%s: %s %s idle\n", prog, etime(), c->con_hostname);

            /* If there's a PGP session key for this host, don't actually
	       release the connection buffer, as that would lose it.  Issue
	       an idle message, though, and tweak the timeout so we do this
	       only once. */

	    if (c->pgpkey[0] == 0) {
		if (l == NULL) {
		    conn = n;
		} else {
		    l->con_next = n;
		}
		free(c);
	    }
	} else {
	    l = c;
	}
	c = n;
    }

    /* Release the sound device after two ticks.  This allows
       other programs to use it while we're idle. */

    if (audiok && ++audiotime >= 2) {
	soundterm();
	audiok = FALSE;
	if (Debug) {
            fprintf(stderr, "%s: releasing audio device.\n", prog);
	}

        /* Flush the record file so if we're rudely terminated it
	   will be up to date as of the last audio idle time. */

	if (record != NULL) {
	    fflush(record);
	}
    }

    /* If we still own the sound device or have a connection
       open, reset the timer. */

    if (audiok || (actives > 0)) {
	alarm(TickTock);	      /* Wind the cat */
	if (Debug) {
            fprintf(stderr, "%s: Tick....\n", prog);
	}
    } else {
	if (Debug) {
            fprintf(stderr, "%s: Tock....\n", prog);
	}
    }
}

/*  PLAYBUFFER	--  Send next buffer to audio output. */

static void playbuffer(msg, c)
  soundbuf *msg;
  struct connection *c;
{
    char *val;
    LONG len;
    char auxbuf[BUFL + 2], bbuf[8], tbuf[8];

    debugging = (msg->compression & fDebug) ? TRUE : FALSE;

    audiotime = 0;		      /* Reset timeout counter */
    if (!audiok) {
	if (!soundinit(O_WRONLY)) {
            perror("opening audio output device");
            return;                   /* Can't acquire sound device */
	}
	audiok = TRUE;
	if (Debug) {
            fprintf(stderr, "%s: opening audio device.\n", prog);
	}
	signal(SIGALRM, release);     /* Set signal to handle timeout */
	alarm(TickTock);	      /* Set alarm clock to free audio device */
    }

    if (Debug) {
        fprintf(stderr, "%s: playing %d%s bytes from %s.\n", prog, msg->buffer.buffer_len,
		((msg->compression & (fComp2X | fCompGSM)) ==
				     (fComp2X | fCompGSM)) ?
                                     " GSM+2X compressed" :
		((msg->compression & (fComp2X | fCompADPCM)) ==
				     (fComp2X | fCompADPCM)) ?
                                     " ADPCM+2X compressed" :
                ((msg->compression & fCompADPCM) ? " ADPCM compressed" :
                ((msg->compression & fComp2X) ? " 2X compressed" :
                ((msg->compression & fCompGSM) ? " GSM compressed" :  ""))),
		msg->sendinghost);
    }
    len = msg->buffer.buffer_len;
    val = msg->buffer.buffer_val;

    /* If the fSetDest bit is on, use the fDestJack bit to re-route
       the sound.  This is normally used to divert the sound to the
       speaker to get an individual's attention. */

    if (msg->compression & fSetDest) {
	sounddest((msg->compression & fDestJack) ? 1 : 0);
	if (!(msg->compression & fDestJack)) {
	    soundplayvol(50);	      /* Make sure volume high enough */
	}
    }

    /* If message is encrypted, decrypt. */

    if ((msg->compression & fEncOTP) && (curotp != NULL)) {
	int i;

	if (Debug) {
            fprintf(stderr, "%s: decrypting %d bytes with key file.\n", prog, len);
	}
	for (i = 0; i < len; i ++) {
	    val[i] ^= curotp[i];
	}
    }

    if ((msg->compression & fEncPGP) && c->pgpkey[0]) {
	unsigned short iv[4];
	LONG slen = (len + 7) & (~7);

	bzero(iv, sizeof(iv));
	initcfb_idea(iv, c->pgpkey + 1, TRUE);

	if (Debug) {
            fprintf(stderr, "%s: decrypting %d bytes with PGP key.\n", prog, slen);
	}
	ideacfb(val, slen);
	close_idea();
    }

    if ((msg->compression & fEncIDEA) && ideakey[0]) {
	unsigned short iv[4];
	LONG slen = (len + 7) & (~7);

	bzero(iv, sizeof(iv));
	initcfb_idea(iv, ideakey + 1, TRUE);

	if (Debug) {
            fprintf(stderr, "%s: decrypting %d bytes with IDEA key.\n", prog, slen);
	}
	ideacfb(val, slen);
	close_idea();
    }

    if ((msg->compression & fEncDES) && deskey[0]) {
	int i;

	setkey(deskey + 1);

	if (Debug) {
            fprintf(stderr, "%s: decrypting %d bytes with DES key.\n", prog, len);
	}
	for (i = 0; i < len; i += 8) {
	    bcopy(val + i, tbuf, 8);
	    dedes(val + i);

	    /* Reverse cipher block chaining. */

	    if (i > 0) {
		int j;

		for (j = 0; j < 8; j++) {
		    val[(i + j)] ^= bbuf[j];
		}
	    }
	    bcopy(tbuf, bbuf, 8);
	}
    }

    /* If message is compressed, decompress appropriately. */

    if (msg->compression & fCompGSM) {
	gsmdecomp(msg);
	len = msg->buffer.buffer_len;
    }

    if (msg->compression & fCompADPCM) {
	adpcmdecomp(msg);
	len = msg->buffer.buffer_len;
    }

    if (msg->compression & fComp2X) {
	int i;
	register char *ab = auxbuf;

	assert(len < sizeof auxbuf);
	for (i = 0; i < len; i++) {
	    *ab++ = i == 0 ? *val :
#define SMOOTH_2X
#ifdef SMOOTH_2X
		(audio_s2u((audio_u2s(*val) + audio_u2s(val[-1])) / 2));   
#else
		val[-1];
#endif
	    *ab++ = *val++;
	}
	len *= 2;
	val = auxbuf;
    }

    soundplay(len, val);
    if (record != NULL) {
	fwrite(val, len, 1, record);
    }
}

/*  USAGE  --  Print how-to-call information.  */

static void usage()
{
    V fprintf(stderr, "%s  --  Speak Freely sound receiver.\n", prog);
    V fprintf(stderr, "               Release 5.3, September 1995.\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "Usage: %s [options]\n", prog);
    V fprintf(stderr, "Options:\n");
    V fprintf(stderr, "           -D         Force debug output\n");
    V fprintf(stderr, "           -Ikey      IDEA decrypt with key\n");
    V fprintf(stderr, "           -Kkey      DES decrypt with key\n");
#ifdef MULTICAST
    V fprintf(stderr, "           -Mhost/ip  Join multicast to given name or IP address\n");
#endif
    V fprintf(stderr, "           -Ofile     Use file as key file\n");
    V fprintf(stderr, "           -Q         Prevent debug output\n");
    V fprintf(stderr, "           -R[+]file  Record [append] sound in file\n");
    V fprintf(stderr, "           -U         Print this message\n");
    V fprintf(stderr, "           -Vtimeout  Show hostnames that connect\n");
    V fprintf(stderr, "           -Z\"phrase\" Set PGP secret key pass phrase\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "by John Walker\n");
    V fprintf(stderr, "   E-mail: kelvin@fourmilab.ch\n");
    V fprintf(stderr, "   WWW:    http://www.fourmilab.ch/\n");
}

/*  Main program.  */

main(argc, argv)
  int argc;
  char *argv[];
{
    int i, j, k, l, sock, length;
    struct sockaddr_in name;
    struct soundbuf sb;
    FILE *fp;
    struct connection *c;
    int newconn;
    struct auhdr {		      /* .au file header */
	char magic[4];
	long hsize, dsize, emode, rate, nchan;
    };
    static struct auhdr afh = { ".snd", sizeof(struct auhdr), 1, 1, 8000, 1 };

    /* Create the socket from which to read */

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0) {
        perror("opening datagram socket");
	return 1;
    }

    /* Create name with wildcards. */

    name.sin_family = AF_INET;
    name.sin_addr.s_addr = INADDR_ANY;
    name.sin_port = htons(Internet_Port);
    if (bind(sock, (struct sockaddr *) &name, sizeof name) < 0) {
        perror("binding datagram socket");
	return 1;
    }

    /*	Process command line options.  */

    pgppass = getenv("PGPPASS");
    prog = argv[0];
    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

                case 'D':             /* -D  --  Force debug output */
		    debugforce = 1;
		    break;

                case 'I':             /* -Ikey  --  Set IDEA key */
		    if (strlen(op + 1) == 0) {
			ideakey[0] = FALSE;
		    } else {
			struct MD5Context md5c;

			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(ideakey + 1, &md5c);
			ideakey[0] = TRUE;
		    }
		    break;

                case 'K':             /* -Kkey  --  Set DES key */
		    desinit(1);       /* Initialise the DES library */
		    if (strlen(op + 1) == 0) {
			deskey[0] = FALSE;
		    } else {
			struct MD5Context md5c;
			char md5key[16];

			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(md5key, &md5c);
			for (j = 0; j < 8; j++) {
			    deskey[j + 1] = (char)
					  ((md5key[j] ^ md5key[j + 8]) & 0x7F);
			}
			deskey[0] = TRUE;
		    }
		    break;

#ifdef MULTICAST
                case 'M':             /* -Mhost/ip -- Join multicast to name/IP number */
		    {
			struct ip_mreq m;
			struct hostent *h;
			long naddr;

                        /* If it's a valid IP number, use it.  Otherwise try to look
			   up as a host name. */

			if ((naddr = inet_addr(op + 1)) == -1) {
			    h = gethostbyname(op + 1);
			    if (h == 0) {
                                fprintf(stderr, "%s: unknown multicast group\n", op + 1);
				return 2;
			    }
			    bcopy((char *) h->h_addr, (char *) &naddr, sizeof naddr);
			}
			m.imr_multiaddr.s_addr = naddr;
			m.imr_interface.s_addr = htons(INADDR_ANY);
			if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP,
			    &m, sizeof m) < 0) {
                            perror("joining multicast group");
			    return 2;
			}
		    }
		    break;
#endif

                case 'O':             /* -Ofile -- Use file as key file */
                    fp = fopen(op + 1, "r");
		    if (fp == NULL) {
                        perror("Cannot open key file");
			return 2;
		    }
		    curotp = malloc(BUFL);
		    if (curotp == NULL) {
                        fprintf(stderr, "%s: Cannot allocate key file buffer.\n", prog);
			return 2;
		    }
		    l = fread(curotp, 1, BUFL, fp);
		    if (l == 0) {
                        /* Idiot supplied void key file.  Give 'im
			   what he asked for: no encryption. */
			curotp[0] = 0;
			l = 1;
		    }
		    fclose(fp);
		    /* If the file is shorter than the maximum buffer
		       we may need to encrypt, replicate the key until
		       the buffer is filled. */
		    j = l;
		    k = 0;
		    while (j < BUFL) {
			curotp[j++] = curotp[k++];
			if (k >= l) {
			    k = 0;
			}
		    }
		    break;

                case 'Q':             /* -Q  --  Prevent debug output */
		    debugforce = -1;
		    break;

                case 'R':             /* -Rfile  --  Answering machine record */
		    {
			char *fn = op + 1;
			int append = FALSE;

                        if (*fn == '+') {
			    fn++;
			    append = TRUE;
			}
                        record = fopen(fn, append ? "a" : "w");
			if (record == NULL) {
                            perror("Cannot create answering machine file");
			    return 2;
			}
			fseek(record, 0L, 2);
			if (ftell(record) == 0) {
			    fwrite(&afh, sizeof afh, 1, record);
			}
			fflush(record);
		    }
		    break;

                case 'U':             /* -U  --  Print usage information */
                case '?':             /* -?  --  Print usage information */
		    usage();
		    return 0;

                case 'V':             /* -V  --  Show hostnames that connect */
		    showhosts = TRUE;
		    if (op[1] != 0) {
			int t = atoi(op + 1);

			if (t > 0) {
			    if (t < (TickTock + 1)) {
				t = TickTock + 1;
			    }
			    hosttimeout = (t / TickTock) * TickTock;
			}
		    }
		    break;

                case 'Z':             /* -Z"phrase"  --  Set PGP pass phrase */
		    if (op[1] != 0) {
			pgppass = strdup(op + 1);
			op = argv[i];
                        /* Zap the pass phrase in memory so "ps" etc. don't see it. */
			while (*op != 0) {
                            *op++ = ' ';
			}
		    } else {
			char s[256];

                        fprintf(stderr, "PGP secret key pass phrase: ");
			fflush(stderr);
			initscr();
			noecho();
			getstr(s);
			pgppass = strdup(s);
			echo();
			endwin();
                        fprintf(stderr, "\n");
		    }
		    break;
	    }
	} else {
	    usage();
	    return 2;
	}
    }

#ifdef HALF_DUPLEX
    {
	struct hostent *h;
	char host[512];

	gethostname(host, sizeof host);
	h = gethostbyname(host);
	bcopy((char *) (h->h_addr), (char *) &localhost,
	    sizeof localhost);
#ifdef HDX_DEBUG
	if (Debug) {
            fprintf(stderr, "%s: local host %s = %s\n", prog, host,
		   inet_ntoa(localhost));
	}
#endif
    }
#endif

    /* Initialise GSM decoding */

    gsmh = gsm_create();

    /* Find assigned port value and print it. */

    length = sizeof(name);
    if (getsockname(sock, (struct sockaddr *) &name, &length) < 0) {
        perror("getting socket name");
	return 1;
    }
#ifdef SHOW_SOCKET
    fprintf(stderr, "%s: socket port #%d\n", prog, ntohs(name.sin_port));
#endif

    /* Read from the socket. */

    while (TRUE) {
#ifdef SHOW_SOCKET
	int rll;
#endif

	fromlen = sizeof(from);
	if ((
#ifdef SHOW_SOCKET
	     rll =
#endif
		   recvfrom(sock, (char *) &sb, sizeof sb, 0, (struct sockaddr *) &from, &fromlen)) < 0) {
#ifdef sgi
	    if (errno != EINTR) {
                perror("receiving datagram packet");
	    }
#endif
	    continue;
	}
#ifdef SHOW_SOCKET
	if (Debug) {
            fprintf(stderr, "%s: %d bytes read from socket.\n", prog, rll);
	}
#endif

#ifdef HALF_DUPLEX

	/* If this is a half duplex mute request, immediately release
	   the audio output device if we have it.  We verify the
           sender's address and accept mute requests only from the
	   local host; nobody else has any business telling us
	   to shut up! */

	if (sb.compression & fHalfDuplex) {
#ifdef HDX_DEBUG
	    if (Debug) {
                static char *hdxreq[4] = { "Bogus(0)", "Mute", "Resume",
                                           "Bogus(3)" };
		unsigned char *cp = (unsigned char *) &localhost;

                fprintf(stderr, "%s: half-duplex %s request from %s.\n",
		    prog, hdxreq[sb.compression & 3], sb.sendinghost);
	    }
#endif
	    if (memcmp(&from.sin_addr, &localhost, sizeof localhost) == 0) {
		if (sb.compression & fHalfDuplexMute) {
		    if (audiok) {
			soundterm();
			audiok = FALSE;
			if (Debug) {
                            fprintf(stderr, "%s: half-duplex releasing audio output.\n", prog);
			}
		    }
		    halfDuplexMuted = TRUE;
		} else if (sb.compression & fHalfDuplexResume) {
		    halfDuplexMuted = FALSE;
		}
	    }
	    continue;		      /* Done with packet */
	}
#endif

	/* If the packet requests loop-back, immediately dispatch it
	   back to the host who sent it to us.	To prevent an infinite
	   loop-back cycle, we clear the loop-back bit in the header
	   before sending the message.	We leave the host of origin
	   unchanged, allowing the sender to identify the packet as
	   one he originated. */

	if (sb.compression & fLoopBack) {
	    bcopy((char *) &(from.sin_addr), (char *) &(name.sin_addr),
		sizeof(struct in_addr));
	    sb.compression &= ~fLoopBack;    /* Prevent infinite loopback */
	    if (sendto(sock, (char *) &sb,
		(int) ((sizeof(struct soundbuf) - BUFL) + sb.buffer.buffer_len),
		0, (struct sockaddr *) &(name), sizeof name) < 0) {
                perror("sending datagram message");
	   }
	}

	/* See if this connection is active.  If not, initialise a new
	   connection. */

	newconn = FALSE;
	c = conn;
	while (c != NULL) {
	    if (memcmp(&from.sin_addr, &(c->con_addr),
		       sizeof(struct in_addr)) == 0) {
		break;
	    }
	    c = c->con_next;
	}
	if (c == NULL) {
	    c = (struct connection *) malloc(sizeof(struct connection));
	    if (c != NULL) {
		struct hostent *h;

		newconn = TRUE;
		c->con_next = conn;
		c->pgpkey[0] = FALSE;
		bzero(c->keymd5, 16);
		conn = c;
		bcopy(&from.sin_addr, &(c->con_addr),
		    sizeof(struct in_addr));
		h = gethostbyaddr((char *) &from.sin_addr, sizeof(struct in_addr),
				  AF_INET);
		if (h == NULL) {
		    strcpy(c->con_hostname, inet_ntoa(from.sin_addr));
		} else {
		    strcpy(c->con_hostname, h->h_name);
		}
	    }
	} else if (c->con_timeout == (hosttimeout + 1)) {
	    newconn = TRUE;
	}
	if (c != NULL) {
	    /* Reset connection timeout. */
	    c->con_timeout = 0;
	    if (newconn) {
                fprintf(stderr, "%s: %s %s connect\n", prog, etime(), c->con_hostname);
	    }
	}

#ifdef HALF_DUPLEX

        /* If we're muted by a transmission in progress on half-duplex
	   audio hardware, this is the end of line for this sound
	   packet. */

	if (halfDuplexMuted) {
	    if (Debug) {
                fprintf(stderr, "%s: %s packet lost by half-duplex muting.\n",
				prog, c->con_hostname);
	    }
	    continue;
	}
#endif

	if ((sb.compression & fKeyPGP)) {
	    char cmd[256], f[40], kmd[16];
	    FILE *kfile;
	    FILE *pipe;
	    long flen;
	    struct MD5Context md5c;

	    MD5Init(&md5c);
	    MD5Update(&md5c, sb.buffer.buffer_val, sb.buffer.buffer_len);
	    MD5Final(kmd, &md5c);

	    if (memcmp(c->keymd5, kmd, 16) != 0) {
		bcopy(kmd, c->keymd5, 16);
                sprintf(f, "/tmp/.SF_SKEY%d", getpid());

                kfile = fopen(f, "w");
		if (kfile == NULL) {
                    fprintf(stderr, "Cannot open encrypted session key file %s\n", f);
		} else {
		    fwrite(sb.buffer.buffer_val, sb.buffer.buffer_len, 1, kfile);
		    fclose(kfile);
#ifdef ZZZ
		    if (pgppass == NULL) {
			static char s[256]; 

                        fprintf(stderr, "Enter PGP pass phrase: ");
			if (fgets(s, sizeof s, stdin) != NULL) {
			    s[strlen(s) - 1] = 0;
			    pgppass = s;
			}
		    }
#endif
                    sprintf(cmd, "pgp -f +nomanual +verbose=0 %s%s%s <%s",
                        pgppass ? "-z\"" : "", pgppass ? pgppass : "",
                        pgppass ? "\" " : "", f);
#ifdef PGP_DEBUG
                    fprintf(stderr, "Decoding session key with: %s\n", cmd);
#else
		    if (Debug) {
                       fprintf(stderr, "%s: decoding PGP session key.\n", prog);
		    }
#endif
                    pipe = popen(cmd, "r");
		    if (pipe == NULL) {
                        fprintf(stderr, "Unable to open pipe to: %s\n", c);
		    } else {
			if (fread(c->pgpkey, 1, 17, pipe) == 17) {
			    c->pgpkey[0] = TRUE;
#ifdef PGP_DEBUG
			    {	
				int i;

                                fprintf(stderr, "Session key for %s:", c->con_hostname);
				for (i = 0; i < 16; i++) {
                                    fprintf(stderr, " %02X", c->pgpkey[i + 1] & 0xFF);
				}
                                fprintf(stderr, "\n");
			    }
#else
			    if (Debug) {
                               fprintf(stderr, "%s: PGP session key decoded.\n", prog);
			    }
#endif
			} else {
			    c->pgpkey[0] = FALSE;
                            fprintf(stderr, "%s: Error decoding PGP session key.\n", prog);
			}
			pclose(pipe);
		    }
		    unlink(f);
		}
	    }
	} else {
	    playbuffer(&sb, c);
	}
    }
#ifdef MEANS_OF_EXIT
    close(sock);
    desdone();
    gsm_destroy(gsmh);
    if (record != NULL) {
	fclose(record);
    }
    return 0;
#endif
}
