/*

			Speak Freely for Unix
		  Network sound transmission program

	Designed and implemented in July of 1991 by John Walker

*/

#include "speakfree.h"

/*  Destination host descriptor.  */

struct destination {
    struct destination *dnext;	      /* Next destination in list */
    char *server;		      /* Host name identifier string */
    struct sockaddr_in name;	      /* Internet address */
    struct sockaddr_in ctrl;	      /* RTCP port address */
#ifdef RTP_SUPPORT
    int isrtp;			      /* Is this an RTP connection ? */
#endif
    unsigned char time_to_live;       /* Multicast scope specification */
    char deskey[9];		      /* Destination DES key, if any */
    char ideakey[17];		      /* Destination IDEA key, if any */
    char pgpkey[17];		      /* Destination PGP key, if any */
    char *otp;			      /* Key file */
};

static char *progname;		      /* Program name */
static int sock;		      /* Communication socket */
static struct destination *dests = NULL, *dtail;
static int compressing = FALSE;       /* Compress sound: simple 2X */
static int gsmcompress = TRUE;	      /* GSM compress buffers */
static int lpccompress = FALSE;       /* LPC compress buffers */
static int adpcmcompress = FALSE;     /* ADPCM compress buffers */
static int squelch = 0; 	      /* Squelch level if > 0 */
#ifdef PUSH_TO_TALK
static int push = TRUE; 	      /* Push to talk mode */
static int talking = FALSE;	      /* Push to talk button state */
static int rawmode = FALSE;	      /* Is terminal in raw mode ? */
#endif
static int ring = FALSE;	      /* Force speaker & level on next pkt ? */
#ifdef RTP_SUPPORT
static int rtp = FALSE; 	      /* Use Internet Real-Time Protocol */
#endif
static int agc = FALSE; 	      /* Automatic gain control active ? */
static int rgain = 33;		      /* Current recording gain level */
static int debugging = FALSE;	      /* Debugging enabled here and there ? */
static int havesound = FALSE;         /* True if we've acquired sound input */
static char hostname[20];	      /* Host name to send with packets */
static int loopback = FALSE;	      /* Remote loopback mode */
static gsm gsmh;		      /* GSM handle */
static struct adpcm_state adpcm;      /* ADPCM compression state */

static unsigned long ssrc;	      /* RTP synchronisation source identifier */
static unsigned long timestamp;       /* RTP packet timestamp */
static unsigned short seq;	      /* RTP packet sequence number */

static char *sdes = NULL;	      /* RTP SDES packet */
static int sdesl;		      /* RTP SDES packet length */

static char curkey[9] = "";           /* Current DES key if curkey[0] != 0 */
static char curideakey[17] = "";      /* Current IDEA key if curideakey[0] != 0 */
static char curpgpkey[17] = "";       /* Current PGP key if curpgpkey[0] != 0 */
static char *curotp = NULL;	      /* Key file buffer */
#ifdef TINY_PACKETS
static int sound_packet;	      /* Ideal samples/packet */
#endif
static struct soundbuf *pgpsb = NULL; /* PGP key sound buffer, if any */
static LONG pgpsbl;		      /* Length of PGP key sound buffer data */
#ifdef HALF_DUPLEX
static struct in_addr localhost;      /* Our internet address */
static int halfDuplexMuted = FALSE;   /* Muted by half-duplex transmission */
#endif
static int hasFace = FALSE;	      /* Is a face image available ? */

#ifdef sgi
static long usticks;		      /* Microseconds per clock tick */
#endif

#define ucase(x)    (islower(x) ? toupper(x) : (x))

/*  GSMCOMP  --  Compress the contents of a sound buffer using GSM.  */

static void gsmcomp(sb)
  struct soundbuf *sb;
{
    gsm_signal src[160];
    gsm_frame dst;
    int i, j, l = 0;
    char *dp = ((char *) sb->buffer.buffer_val) + sizeof(short);

    sb->compression |= fCompGSM;
    for (i = 0; i < sb->buffer.buffer_len; i += 160) {
	for (j = 0; j < 160; j++) {
	    if ((i + j) < sb->buffer.buffer_len) {
		src[j] = audio_u2s(sb->buffer.buffer_val[i + j]);
	    } else {
		src[j] = 0;
	    }
	}
	gsm_encode(gsmh, src, dst);
	bcopy(dst, dp, sizeof dst);
	dp += sizeof dst;
	l += sizeof dst;
    }

    /* Hide original uncompressed buffer length in first 2 bytes of buffer. */

    *((short *) sb->buffer.buffer_val) = sb->buffer.buffer_len;
    sb->buffer.buffer_len = l + sizeof(short);
}

/*  ADPCMCOMP  --  Compress the contents of a sound buffer using ADPCM.  */

static void adpcmcomp(sb)
  struct soundbuf *sb;
{
    unsigned char *dp = (unsigned char *) sb->buffer.buffer_val;
    struct adpcm_state istate;

    istate = adpcm;
    sb->compression |= fCompADPCM;
    adpcm_coder_u(dp, (char *) dp, sb->buffer.buffer_len, &adpcm);
    sb->buffer.buffer_len /= 2;

    /* Hide the ADPCM encoder state at the end of this buffer.
       The shifting and anding makes the code byte-order
       insensitive. */

    dp += sb->buffer.buffer_len;
    *dp++ = ((unsigned int) istate.valprev) >> 8;
    *dp++ = istate.valprev & 0xFF;
    *dp = istate.index;
    sb->buffer.buffer_len += 3;
}

/*  LPCCOMP  --  Compress the contents of a sound buffer using LPC.  */

static void lpccomp(sb)
  struct soundbuf *sb;
{
    int i, l = 0;
    char *dp = ((char *) sb->buffer.buffer_val) + sizeof(short);
    unsigned char *src = ((unsigned char *) sb->buffer.buffer_val);
    lpcparams_t lp;

    sb->compression |= fCompLPC;
    for (i = 0; i < sb->buffer.buffer_len; i += 180) {
	lpc_analyze(src + i, &lp);
	bcopy(&lp, dp, sizeof lp);
	dp += sizeof lp;
	l += sizeof lp;
    }

    /* Hide original uncompressed buffer length in first 2 bytes of buffer. */

    *((short *) sb->buffer.buffer_val) = sb->buffer.buffer_len;
    sb->buffer.buffer_len = l + sizeof(short);
}

/*  ADDEST  --	Add destination host to host list.  */

static int addest(host)
  char *host;
{
    struct destination *d;
    struct hostent *h;
    long naddr;
    unsigned int ttl = 1;
    char *mcs;
    int curport = Internet_Port;

    /* If a multicast scope descriptor appears in the name, scan
       it and lop it off the end.  We'll apply it later if we discover
       this is actually a multicast address. */

    if ((mcs = strchr(host, '/')) != NULL) {
	*mcs++ = 0;
#ifdef MULTICAST
	ttl = atoi(mcs);
#endif
    }

    /* If a port number appears in the name, scan it and lop
       it off. */

    if ((mcs = strchr(host, ':')) != NULL) {
	*mcs++ = 0;
	curport = atoi(mcs);
    }

    /* If it's a valid IP number, use it.  Otherwise try to look
       up as a host name. */

    if ((naddr = inet_addr(host)) != -1) {
    } else {
	h = gethostbyname(host);
	if (h == 0) {
            fprintf(stderr, "%s: unknown host\n", host);
	    return FALSE;
	}
	bcopy((char *) h->h_addr, (char *) &naddr, sizeof naddr);
    }

#ifdef MULTICAST
    if (!IN_MULTICAST(naddr)) {
	ttl = 0;
    }
#endif

    d = (struct destination *) malloc(sizeof(struct destination));
    d->dnext = NULL;
    d->server = host;
    bcopy((char *) &naddr, (char *) &(d->name.sin_addr), sizeof naddr);
    bcopy((char *) &naddr, (char *) &(d->ctrl.sin_addr), sizeof naddr);
    bcopy(curkey, d->deskey, 9);
    bcopy(curideakey, d->ideakey, 17);
    bcopy(curpgpkey, d->pgpkey, 17);
    d->otp = curotp;
    d->time_to_live = ttl;
#ifdef RTP_SUPPORT
    d->isrtp = rtp;
#endif
    d->name.sin_family = AF_INET;
    d->name.sin_port = htons(curport);
    d->ctrl.sin_family = AF_INET;
    d->ctrl.sin_port = htons(curport + 1);
    if (dests == NULL) {
	dests = d;
    } else {
	dtail->dnext = d;
    }
    dtail = d;

#ifdef RTP_SUPPORT
    if (d->isrtp) {
	if (sendto(sock, sdes, sdesl,
	     0, (struct sockaddr *) &(d->ctrl), sizeof d->ctrl) < 0) {
            perror("sending RTCP packet");
	    return FALSE;
	}
    }
#endif

    if (pgpsb != NULL) {
	int i;

        /* "What I tell you three times is true".  When we change
	   over to a TCP connection bracketing the UDP sound data
	   transmission, we can send this just once, knowing it has
	   arrived safely. */

	for (i = 0; i < 3; i++) {
	    if (sendto(sock, (char *) pgpsb, (sizeof(struct soundbuf) - BUFL) + pgpsbl,
		0, (struct sockaddr *) &(d->name), sizeof d->name) < 0) {
                perror("sending PGP session key");
		break;
	    }
	}
/* No need to sleep once ack from far end confirms key decoded. */
sleep(7);
    }
    return TRUE;
}

/*  SENDPKT  --  Send a message to all active destinations.  */

static int sendpkt(sb)
  struct soundbuf *sb;
{
    struct destination *d;

    if (gsmcompress) {
	gsmcomp(sb);
    }

    if (adpcmcompress) {
	adpcmcomp(sb);
    }

    if (lpccompress) {
	lpccomp(sb);
    }

    if (hasFace) {
	sb->compression |= fFaceOffer;
    }

    for (d = dests; d != NULL; d = d->dnext) {
#ifdef MULTICAST
	if (IN_MULTICAST(d->name.sin_addr.s_addr)) {
	    setsockopt(sock, IPPROTO_IP, IP_MULTICAST_TTL,
		&(d->time_to_live), sizeof d->time_to_live);
	    if (debugging) {
                fprintf(stderr, "Multicasting with scope of %d to %s.\n",
		    d->time_to_live, inet_ntoa(d->name.sin_addr));
	    }
	}
#endif
	if (d->deskey[0] || d->ideakey[0] || d->pgpkey[0] || d->otp != NULL) {
	    soundbuf ebuf;
	    int i;
	    LONG slen;

	    bcopy(sb, &ebuf, (sizeof(struct soundbuf) - BUFL) + sb->buffer.buffer_len);
	    slen = ebuf.buffer.buffer_len;

	    /* DES encryption. */

	    if (d->deskey[0]) {
		setkey(d->deskey + 1);

                /* If we're DES encrypting we must round the size of
		   the data to be sent to be a multiple of 8 so that
		   the entire DES frame is sent. */

		slen = (slen + 7) & (~7);
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with DES key.\r\n",
			    slen);
		}
		for (i = 0; i < ebuf.buffer.buffer_len; i += 8) {

		    /* Apply cipher block chaining within the packet. */

		    if (i > 0) {
			int j;

			for (j = 0; j < 8; j++) {
			    ebuf.buffer.buffer_val[(i + j)] ^=
				ebuf.buffer.buffer_val[(i + j) - 8];
			}
		    }
		    endes(ebuf.buffer.buffer_val + i);
		}
		ebuf.compression |= fEncDES;
	    }

	    /* IDEA encryption. */

	    if (d->ideakey[0]) {
		unsigned short iv[4];

		bzero(iv, sizeof(iv));
		initcfb_idea(iv, d->ideakey + 1, FALSE);

                /* If we're IDEA encrypting we must round the size of
		   the data to be sent to be a multiple of 8 so that
		   the entire IDEA frame is sent. */

		slen = (slen + 7) & (~7);
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with IDEA key.\r\n",
			    slen);
		}
		ideacfb(ebuf.buffer.buffer_val, slen);
		close_idea();
		ebuf.compression |= fEncIDEA;
	    }

	    /* PGP encryption. */

	    if (d->pgpkey[0]) {
		unsigned short iv[4];

		bzero(iv, sizeof(iv));
		initcfb_idea(iv, d->pgpkey + 1, FALSE);

                /* If we're PGP IDEA encrypting we must round the size of
		   the data to be sent to be a multiple of 8 so that
		   the entire IDEA frame is sent. */

		slen = (slen + 7) & (~7);
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with PGP key.\r\n",
			    slen);
		}
		ideacfb(ebuf.buffer.buffer_val, slen);
		close_idea();
		ebuf.compression |= fEncPGP;
	    }

	    /* Key file encryption. */

	    if (d->otp != NULL) {
		if (debugging) {
                    fprintf(stderr, "Encrypting %d bytes with key file.\r\n",
			    ebuf.buffer.buffer_len);
		}
		for (i = 0; i < ebuf.buffer.buffer_len; i ++) {
		    ebuf.buffer.buffer_val[i] ^= d->otp[i];
		}
		ebuf.compression |= fEncOTP;
	    }

	    ebuf.compression = htonl(ebuf.compression);
	    ebuf.buffer.buffer_len = htonl(ebuf.buffer.buffer_len);
	    if (sendto(sock, (char *) &ebuf, (sizeof(struct soundbuf) - BUFL) + slen,
		0, (struct sockaddr *) &(d->name), sizeof d->name) < 0) {
                perror("sending datagram message");
		return FALSE;
	    }
	} else {
	    int pktlen = (int) ((sizeof(struct soundbuf) - BUFL) +
				    sb->buffer.buffer_len);

#ifdef RTP_SUPPORT
	    if (d->isrtp) {
		LONG bufl = sb->buffer.buffer_len;

		if (sendto(sock, sdes, sdesl,
		    0, (struct sockaddr *) &(d->ctrl), sizeof d->ctrl) < 0) {
                    perror("sending RTCP packet");
		    return FALSE;
		}
		pktlen = rtpout(sb, ssrc, timestamp, seq);
		seq++;
		timestamp += sound_packet;
	    } else {
#endif
		sb->compression = htonl(sb->compression);
		sb->buffer.buffer_len = htonl(sb->buffer.buffer_len);
#ifdef RTP_SUPPORT
	    }
#endif
	    if (sendto(sock, (char *) sb, pktlen,
		0, (struct sockaddr *) &(d->name), sizeof d->name) < 0) {
                perror("sending datagram message");
		return FALSE;
	    }
	}
    }
    return TRUE;
}

/*  GETAUDIO  --  Open audio input.  If audio hardware is half duplex,
		  we may have to mute output in progress.  */

static int getaudio()
{
    int i;

#ifdef HALF_DUPLEX
    struct soundbuf hdreq;
    struct sockaddr_in name;

    /* If we're afflicted with half-duplex hardware, make several
       attempts to mute the output device and open audio input.
       We send a mute even if the output isn't active as a courtesy
       to the output program; if a later attempt to acquire output
       fails, he'll know it was a result of being muted. */

    for (i = 5; i > 0; i--) {
	hdreq.compression = htonl(fHalfDuplex | fHalfDuplexMute);
	strcpy(hdreq.sendinghost, hostname);
	hdreq.buffer.buffer_len = 0;

	bcopy((char *) &localhost, (char *) &(name.sin_addr), sizeof localhost);
	name.sin_family = AF_INET;
	name.sin_port = htons(Internet_Port);
	if (debugging) {
            fprintf(stderr,"Sending fHalfDuplex | fHalfDuplexMute Request\n");
	}
	if (sendto(sock, &hdreq, (sizeof(struct soundbuf) - BUFL),
		   0, (struct sockaddr *) &name, sizeof name) < 0) {
            perror("sending half-duplex mute request");
	    break;
	}
#if defined(BSD_like) || defined(Solaris)
	 usleep(50000L);
#endif
#ifdef sgi
	 sginap(50000L / usticks);
#endif
	if (soundinit(O_RDONLY)) {
	    halfDuplexMuted = TRUE;
	    break;
	}
    }

    /* If we failed to initialise, send a resume just in case
       one of our mute requests made it through anyway. */

    if (i <= 0) {
	hdreq.compression = htonl(fHalfDuplex | fHalfDuplexResume);
	if (sendto(sock, &hdreq, (sizeof(struct soundbuf) - BUFL),
		   0, (struct sockaddr *) &name, sizeof name) < 0) {
            perror("sending half-duplex resume request");
	}
    }
#else
    i = soundinit(O_RDONLY);
#endif

    if (i <= 0) {
            fprintf(stderr, "%s: unable to initialise audio.\n", progname);
	    return FALSE;
    }
    havesound = TRUE;
    if (agc) {
	soundrecgain(rgain);	  /* Set initial record level */
    }
    return TRUE;
}

/*  FREEAUDIO  --  Release sound input device, sending a resume to
                   the output device if we've muted it due to
		   half-duplex hardware.  */

static void freeaudio()
{
    if (havesound) {
	if (debugging) {
            fprintf(stderr, "Restoring audio modes at release.\n");
	}
	soundterm();
	havesound = FALSE;
    }

#ifdef HALF_DUPLEX

    /* When bailing out, make sure we don't leave the output
       muted. */

    if (halfDuplexMuted) {
	struct soundbuf hdreq;
	struct sockaddr_in name;

	hdreq.compression = htonl(fHalfDuplex | fHalfDuplexResume);
	strcpy(hdreq.sendinghost, hostname);
	hdreq.buffer.buffer_len = 0;

	bcopy((char *) &localhost, (char *) &(name.sin_addr), sizeof localhost);
	name.sin_family = AF_INET;
	name.sin_port = htons(Internet_Port);
	if (sendto(sock, &hdreq, (sizeof(struct soundbuf) - BUFL),
		   0, (struct sockaddr *) &name, sizeof name) < 0) {
            perror("sending half-duplex resume request");
	}
	halfDuplexMuted = FALSE;
    }
#endif
}

/*  EXITING  --  Catch as many program termination signals as
		 possible and restore initial audio modes before
		 we exit.  */

static void exiting()
{
    struct destination *d;

#ifdef PUSH_TO_TALK
    if (rawmode) {
        fprintf(stderr, "\r      \r");
	fcntl(fileno(stdin), F_SETFL, 0);
	nocbreak();
	echo();
	endwin();
    }
#endif

#ifdef RTP_SUPPORT
    for (d = dests; d != NULL; d = d->dnext) {
	if (d->isrtp) {
	    char v[1024];
	    int l;

            l = rtp_make_bye(v, ssrc, "Exiting Speak Freely");
	    if (sendto(sock, v, l,
		0, (struct sockaddr *) &(d->ctrl), sizeof d->ctrl) < 0) {
                perror("sending RTCP BYE packet");
	    }
	}
    }
#endif

    freeaudio();
    exit(0);
}

/*  TERMCHAR  --  Check for special characters from console when
		  in raw mode.	*/

static void termchar(ch)    
  int ch;
{
#define Ctrl(x) ((x) - '@')
    if (ch == 'q' || ch == 'Q' || ch == 27 ||
        ch == Ctrl('C') || ch == Ctrl('D')) {
	exiting();
    }
}

/*  SENDFILE  --  Send a file or, if the file name is NULL or a
		  single period, send real-time sound input. */

static int sendfile(f)
  char *f;
{ 
    soundbuf netbuf;
#define buf netbuf.buffer.buffer_val
    int bread, nread;
    FILE *afile = NULL;

#ifdef TINY_PACKETS

    /* Compute the number of sound samples needed to fill a
       packet of TINY_PACKETS bytes. */

    sound_packet = ((TINY_PACKETS - ((sizeof(soundbuf) - BUFL))) *
		    (compressing ? 2 : 1));
    if (gsmcompress) {
	sound_packet = compressing ? 3200 : 1600;
    } else if (adpcmcompress) {
	sound_packet *= 2;
	sound_packet -= 4;	      /* Leave room for state at the end */
    } else if (lpccompress) {
	sound_packet = 1800;
    }

#ifdef RTP_SUPPORT
/* 20 millisecond packets is an explicit RTP standard, like it or not. */
if (rtp) {
    sound_packet = 160;
}
#endif
#ifdef SHOW_PACKET_SIZE
    printf("Samples per packet = %d\n", sound_packet);
#endif
#endif

    strcpy(netbuf.sendinghost, hostname);
    if (f != NULL && (strcmp(f, ".") != 0)) {
	char magic[4];

        afile = fopen(f, "r");
	if (afile == NULL) {
            fprintf(stderr, "Unable to open sound file %s.\n", f);
	    return 2;
	}

	/* If the file has a Sun .au file header, skip it.
	   Note that we still blithely assume the file is
	   8-bit ISDN u-law encoded at 8000 samples per
	   second. */

	fread(magic, 1, sizeof(long), afile);
        if (bcmp(magic, ".snd", 4) == 0) {
	    long startpos;

	    fread(&startpos, sizeof(long), 1, afile);
	    fseek(afile, ntohl(startpos), 0);
	} else {
	    fseek(afile, 0L, 0);
	}
    }

    /* Send a file */

    if (afile) {
#ifdef Solaris
	struct timeval t1, t2;
#endif
#ifdef BSD_like
	struct timeb t1, t2;
#endif
	long et;

	while (
#ifdef Solaris
		    gettimeofday(&t1),
#endif
#ifdef BSD_like
		    ftime(&t1),
#endif
		    (bread = nread =
		    fread(buf, 1,
#ifdef TINY_PACKETS
				  sound_packet,
#else
				  (gsmcompress ? 1600 : BUFL),
#endif
				  afile)) > 0) {
	    netbuf.compression = FALSE | (ring ? (fSetDest | fDestSpkr) : 0);
	    ring = FALSE;
	    netbuf.compression |= debugging ? fDebug : 0;
	    netbuf.compression |= loopback ? fLoopBack : 0;
	    if (compressing) {
		int i;

		nread /= 2;
		for (i = 1; i < nread; i++) {
		    buf[i] = buf[i * 2];
		}
		netbuf.compression |= fComp2X;
	    }
	    netbuf.buffer.buffer_len = nread;
	    if (!sendpkt(&netbuf)) {
		fclose(afile);
		return 1;
	    }

            /* The following code is needed because when we're reading
	       sound from a file, as opposed to receiving it in real
	       time from the CODEC, we must meter out the samples
	       at the rate they will actually be played by the destination
	       machine.  For a 8000 samples per second, this amounts
	       to 125 microseconds per sample, minus the time we spent
	       compressing the data (which is substantial for GSM) and
	       a fudge factor, kOverhead, which accounts for the time
	       spent in executing the delay itself and getting control
               back after it's done.  If sound files pause periodically
               (when the sending machine isn't loaded), you may need
               to reduce the delay parameters.  If they're too low,
	       however, data will be lost when sending long sound files. */

#ifdef TINY_PACKETS
#define kOverhead   8000
#else
#define kOverhead   3000
#endif

#ifdef Solaris
	    gettimeofday(&t2);
	    et = ((bread * 125) -
			(t2.tv_sec - t1.tv_sec) * 1000000 +
			   (t2.tv_usec - t1.tv_usec)) - kOverhead;
#endif
#ifdef BSD_like
	    ftime(&t2);
	    et = ((bread * 125) -
		  1000 * (((t2.time - t1.time) * 1000) +
			   (t2.millitm - t1.millitm))) - kOverhead;
#endif

#ifdef sgi
	    et = (bread * 125) - kOverhead;
#endif

	    if (et > 0) {
#if defined(BSD_like) || defined(Solaris)
		usleep(et);
#endif
#ifdef sgi
		sginap(et / usticks);
#endif
	    }
	    if (debugging) {
                fprintf(stderr, "Sent %d samples from %s in %d bytes.\r\n",
			bread, f, netbuf.buffer.buffer_len);
	    }
	}

	if (debugging) {
            fprintf(stderr, "Sent sound file %s.\r\n", f);
	}
	fclose(afile);
    } else {

	    /* Send real-time audio. */

#ifdef PUSH_TO_TALK
	if (push) {
	    char c;

	    fprintf(stderr,
                "Space bar switches talk/pause, Esc or \"q\" to quit\nPause: ");
	    fflush(stderr);
	    initscr();
	    noecho();
	    cbreak();
	    rawmode = TRUE;
	    read(fileno(stdin), &c, 1);
	    termchar(c);
            fprintf(stderr, "\rTalk:  ");
	    fcntl(fileno(stdin), F_SETFL, O_NDELAY);
	    talking = TRUE;
	}
#endif

	/* Send real-time sound. */

#ifdef Solaris
	setaubufsize(
#ifdef TINY_PACKETS
	    sound_packet
#else
	    sizeof buf
#endif
		    );
#endif

	if (!getaudio()) {
            fprintf(stderr, "Unable to initialise audio.\n");
	    return 2;
	}
	signal(SIGHUP, exiting);     /* Set signal to handle termination */
	signal(SIGINT, exiting);     /* Set signal to handle termination */
	signal(SIGTERM, exiting);    /* Set signal to handle termination */
	while (TRUE) {
	    int soundel = 0;
	    unsigned char *bs = (unsigned char *) buf;

	    if (havesound) {
		soundel = soundgrab(buf,
#ifdef TINY_PACKETS
					 sound_packet
#else
					 sizeof buf
#endif
				   );
	    }

#ifdef PUSH_TO_TALK
	    if (push) {
		char c;

		if (read(fileno(stdin), &c, 1) > 0) {
		    termchar(c);
		    talking = !talking;
		    fflush(stderr);
#ifdef HALF_DUPLEX

		    /* For half-duplex, acquire and release the
		       audio device at each transition.  This lets
		       us mute the output only while in Talk mode. */

		    if (talking) {
			if (!getaudio()) {
                            fprintf(stderr, "Audio device busy.\n");
			    talking = FALSE;
			}
		    } else {
			freeaudio();
		    }
#endif
                    fprintf(stderr, talking ? "\rTalk:   " : "\rPause:  ");
		    fcntl(fileno(stdin), F_SETFL, talking ? O_NDELAY : 0);
		    if (talking) {
			/* Discard all backlog sound input. */
			soundflush();
		    }
		}
	    }
#endif

	    if (soundel > 0) {
		register unsigned char *start = bs;
		register int j;
		int squelched = (squelch > 0), osl = soundel;

		/* If entire buffer is less than squelch, ditch it. */

		if (squelch > 0) {
		    for (j = 0; j < soundel; j++) {
			if (((*start++ & 0x7F) ^ 0x7F) > squelch) {
			    squelched = FALSE;
			    break;
			}
		    }
		}

		if (squelched) {
		    if (debugging) {
                        printf("Entire buffer squelched.\n");
		    }
		} else {
		    netbuf.compression = FALSE | (ring ? (fSetDest | fDestSpkr) : 0);
		    netbuf.compression |= debugging ? fDebug : 0;
		    netbuf.compression |= loopback ? fLoopBack : 0;

		    /* If automatic gain control is enabled,
		       ride the gain pot to fill the dynamic range
		       optimally. */

		    if (agc) {
			register unsigned char *start = bs;
			register int j;
			long msamp = 0;

			for (j = 0; j < soundel; j++) {
			    int s = audio_u2s(*start++);

			    msamp += (s < 0) ? -s : s;
			}
			msamp /= soundel;
			if (msamp < 6000) {
			    if (rgain < 100) {
				soundrecgain(++rgain);
			    }
			} else if (msamp > 7000) {
			    if (rgain > 1) {
				soundrecgain(--rgain);
			    }
			}
		    }

		    ring = FALSE;
		    if (compressing) {
			int i;

			soundel /= 2;
			for (i = 1; i < soundel; i++) {
			    buf[i] = buf[i * 2];
			}
			netbuf.compression |= fComp2X;
		    }
		    netbuf.buffer.buffer_len = soundel;
		    if (!sendpkt(&netbuf)) {
			return 1;
		    }
		    if (debugging) {
                        fprintf(stderr, "Sent %d audio samples in %d bytes.\r\n",
				osl, soundel);
		    }
		}
	    } else {
#if defined(BSD_like) || defined(Solaris)
		usleep(100000L);  /* Wait for some sound to arrive */
#endif
#ifdef sgi
		sginap(100000L / usticks);
#endif
	    }
	}
    }
    return 0;
}

/*  MAKESESSIONKEY  --	Generate session key with optional start
			key.  If mode is TRUE, the key will be
			translated to a string, otherwise it is
			returned as 16 binary bytes.  */

static void makeSessionKey(key, seed, mode)
  char *key, *seed;
  int mode;
{
    int j, k;
    struct MD5Context md5c;
    char md5key[16], md5key1[16];
    char s[1024];

    s[0] = 0;
    if (seed != NULL) {
	strcat(s, seed);
    }

    /* The following creates a seed for the session key generator
       based on a collection of volatile and environment-specific
       information unlikely to be vulnerable (as a whole) to an
       exhaustive search attack.  If one of these items isn't
       available on your machine, replace it with something
       equivalent or, if you like, just delete it. */

    sprintf(s + strlen(s), "%u", getpid());
    sprintf(s + strlen(s), "%u", getppid());
    V getcwd(s + strlen(s), 256);
    sprintf(s + strlen(s), "%u", clock());
    V cuserid(s + strlen(s));
    sprintf(s + strlen(s), "%u", time(NULL));
#ifdef Solaris
    sysinfo(SI_HW_SERIAL,s + strlen(s), 12);
#else
    sprintf(s + strlen(s), "%u", gethostid());
#endif
    getdomainname(s + strlen(s), 256);
    gethostname(s + strlen(s), 256);
    sprintf(s + strlen(s), "%u", getuid());
    sprintf(s + strlen(s), "%u", getgid());
    MD5Init(&md5c);
    MD5Update(&md5c, s, strlen(s));
    MD5Final(md5key, &md5c);
    sprintf(s + strlen(s), "%u", (time(NULL) + 65121) ^ 0x375F);
    MD5Init(&md5c);
    MD5Update(&md5c, s, strlen(s));
    MD5Final(md5key1, &md5c);
    init_idearand(md5key, md5key1, time(NULL));
    if (mode) {
	for (j = k = 0; j < 16; j++) {
	    unsigned char rb = idearand();

#define Rad16(x) ((x) + 'A')
	    key[k++] = Rad16((rb >> 4) & 0xF);
	    key[k++] = Rad16(rb & 0xF);
#undef Rad16
	    if (j & 1) {
                key[k++] = '-';
	    }
	}
	key[--k] = 0;
    } else {
	for (j = 0; j < 16; j++) {
	    key[j] = idearand();
	}
    }
    close_idearand();
}

/*  USAGE  --  Print how-to-call information.  */

static void usage()
{
    V fprintf(stderr, "%s  --  Speak Freely sound sender.\n", progname);
    V fprintf(stderr, "            %s.\n", Release);
    V fprintf(stderr, "\n");
    V fprintf(stderr, "Usage: %s hostname[:port] [options] [ file1 / . ]...\n", progname);
    V fprintf(stderr, "Options: (* indicates defaults)\n");
#ifdef PUSH_TO_TALK
    V fprintf(stderr, "           -A         Always transmit unless squelched\n");
    V fprintf(stderr, "     *     -B         Push to talk using keyboard\n");
#endif
    V fprintf(stderr, "           -C         Compress subsequent sound\n");
    V fprintf(stderr, "           -D         Enable debug output\n");
    V fprintf(stderr, "           -E[key]    Emit session key string\n");
    V fprintf(stderr, "           -F         ADPCM compression\n");
    V fprintf(stderr, "           -G         Automatic gain control\n");
    V fprintf(stderr, "           -Ikey      IDEA encrypt with key\n");
    V fprintf(stderr, "           -Kkey      DES encrypt with key\n");
    V fprintf(stderr, "           -L         Remote loopback\n");
    V fprintf(stderr, "           -LPC       LPC compression\n");
    V fprintf(stderr, "     *     -M         Manual record gain control\n");
    V fprintf(stderr, "           -N         Do not compress subsequent sound\n");
    V fprintf(stderr, "           -Ofile     Use file as key file\n");
    V fprintf(stderr, "           -Phostname[:port] Party line, add host to list\n");
    V fprintf(stderr, "     *     -Q         Disable debug output\n");
    V fprintf(stderr, "           -R         Ring--force volume, output to speaker\n");
#ifdef RTP_SUPPORT
    V fprintf(stderr, "           -RTP       Use Internet Real-Time Protocol\n");
#endif
    V fprintf(stderr, "           -Sn        Squelch at level n (0-255)\n");
    V fprintf(stderr, "     *     -T         Telephone (GSM) compression\n");
    V fprintf(stderr, "           -U         Print this message\n");
    V fprintf(stderr, "           -Z\"user..\" Send PGP session key for user(s)\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "by John Walker\n");
    V fprintf(stderr, "   E-mail: kelvin@fourmilab.ch\n");
    V fprintf(stderr, "   WWW:    http://www.fourmilab.ch/\n");
}

/*  Main program.  */

main(argc, argv)
  int argc;
  char *argv[];
{
    int i, j, k, l, sentfile = 0;
    FILE *fp;
    struct MD5Context md5c;
    char md5key[16], md5key1[16];
    char s[1024];

    progname = argv[0];
    gethostname(hostname, sizeof hostname);
    if (strlen(hostname) > 15) {
	hostname[15] = 0;
    }
    gsmh = gsm_create();
    lpc_init(180);

#ifdef sgi
    usticks = 1000000 / CLK_TCK;
#endif

#ifdef HALF_DUPLEX
    {
	struct hostent *h;
	char host[512];

	gethostname(host, sizeof host);
	h = gethostbyname(host);
	bcopy((char *) (h->h_addr), (char *) &localhost,
	    sizeof localhost);
#ifdef HDX_DEBUG
        fprintf(stderr, "%s: local host %s = %s\n", progname, host,
		inet_ntoa(localhost));
#endif
    }
#endif

    /* Create the socket used to send data. */

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0) {
        perror("opening datagram socket");
	return 1;
    }

    /* See if there's a face image file for this user.  If
       so, we'll offer our face in sound packets we send. */

    {   char *cp = getenv("SPEAKFREE_FACE");
	FILE *facefile;

	if (cp != NULL) {
            if ((facefile = fopen(cp, "r")) == NULL) {
                fprintf(stderr, "%s: cannot open SPEAKFREE_FACE file %s\n",
		    progname, cp);
	    } else {
		fclose(facefile);
		hasFace = TRUE;
	    }
	}
    }

    /*	Process command line options.  */

    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

#ifdef PUSH_TO_TALK
                case 'A':             /* -A  --  Always transmit (no push to talk) */
		    push = FALSE;
		    break;

                case 'B':             /* -B  -- Push to talk (button) */
		    if (isatty(fileno(stdin))) {
			push = TRUE;
		    }
		    break;
#endif

                case 'C':             /* -C  -- Compress sound samples */
#ifdef RTP_SUPPORT
		    if (rtp) {
                        fprintf(stderr, "Simple (-c) compression can't be used in RTP mode.\n");
			return 2;
		    }
#endif
		    compressing = TRUE;
		    break;

                case 'D':             /* -D  --  Enable debug output  */
		    debugging = TRUE;
		    break;

                case 'E':             /* -E  --  Emit session key and exit */
		    makeSessionKey(s, op + 1, TRUE);
                    printf("%s\n", s);
		    return 0;

                case 'F':             /* -F -- ADPCM compression */
		    adpcmcompress = TRUE;
		    gsmcompress = lpccompress = FALSE;
		    break;

                case 'G':             /* -G  --  Automatic gain control */
		    agc = TRUE;
		    break;

                case 'I':             /* -Ikey  --  Set IDEA key */
		    if (strlen(op + 1) == 0) {
			curideakey[0] = FALSE;
		    } else {
			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(curideakey + 1, &md5c);
			curideakey[0] = TRUE;
			if (debugging) {
                            fprintf(stderr, "IDEA key:");
			    for (j = 1; j < 17; j++) {
                                fprintf(stderr, " %02X", (curideakey[j] & 0xFF));
			    }
                            fprintf(stderr, "\n");
			}
		    }
		    break;

                case 'K':             /* -Kkey  --  Set DES key */
		    desinit(1);       /* Initialise the DES library */
		    if (strlen(op + 1) == 0) {
			curkey[0] = FALSE;
		    } else {
			MD5Init(&md5c);
			MD5Update(&md5c, op + 1, strlen(op + 1));
			MD5Final(md5key, &md5c);
			for (j = 0; j < 8; j++) {
			    curkey[j + 1] = (char)
					  ((md5key[j] ^ md5key[j + 8]) & 0x7F);
			}
			if (debugging) {
                            fprintf(stderr, "DES key:");
			    for (j = 0; j < 8; j++) {
                                fprintf(stderr, " %02X", (curkey[j + 1] & 0xFF));
			    }
                            fprintf(stderr, "\n");
			}
			curkey[0] = TRUE;
		    }
		    break;

                case 'L':             /* -L    --  Remote loopback */
				      /* -LPC  --  LPC compress sound */
                    if (ucase(op[1]) == 'P') {
			lpccompress = TRUE;
			gsmcompress = adpcmcompress = FALSE;
		    } else {
			loopback = TRUE;
		    }
		    break;

                case 'M':             /* -M  --  Manual record gain control */
		    agc = FALSE;
		    break;

                case 'N':             /* -N  --  Do not compress sound samples */
		    compressing = FALSE;
		    gsmcompress = FALSE;
		    lpccompress = FALSE;
		    adpcmcompress = FALSE;
		    break;

                case 'O':             /* -Ofile -- Use file as key file */
		    if (op[1] == 0) {
			curotp = NULL; /* Switch off key file */
		    } else {
                        fp = fopen(op + 1, "r");
			if (fp == NULL) {
                            perror("Cannot open key file");
			    return 2;
			}
			curotp = malloc(BUFL);
			if (curotp == NULL) {
                            fprintf(stderr, "Cannot allocate key file buffer.\n");
			    return 2;
			}
			l = fread(curotp, 1, BUFL, fp);
			if (l == 0) {
                            /* Idiot supplied void key file.  Give 'im
			       what he asked for: no encryption. */
			    curotp[0] = 0;
			    l = 1;
			}
			fclose(fp);
			/* If the file is shorter than the maximum buffer
			   we may need to encrypt, replicate the key until
			   the buffer is filled. */
			j = l;
			k = 0;
			while (j < BUFL) {
			    curotp[j++] = curotp[k++];
			    if (k >= l) {
				k = 0;
			    }
			}
		    }
		    break;

                case 'P':             /* -Phost  --  Copy output to host  */
		    if (!addest(op + 1)) {
			return 1;
		    }
		    break;

                case 'Q':             /* -Q  --  Disable debug output  */
		    debugging = FALSE;
		    break;

                case 'R':             /* -R    --  Ring: divert output to speaker */
#ifdef RTP_SUPPORT
				      /* -RTP  --  Use RTP to transmit */
                    if (ucase(op[1]) == 'T') {
			if (compressing) {
                            fprintf(stderr, "Simple (-c) compression can't be used in RTP mode.\n");
			    return 2;
			}
			rtp = TRUE;

			/* Initialise randomised packet counters. */

			makeSessionKey(md5key, NULL, FALSE);
			bcopy(md5key, (char *) &ssrc, sizeof ssrc);
			bcopy(md5key + sizeof ssrc, (char *) &timestamp,
			      sizeof timestamp);
			bcopy(md5key + sizeof ssrc + sizeof timestamp,
			      (char *) &seq, sizeof seq);

			if (sdes == NULL) {
			    sdesl = rtp_make_sdes(&sdes, ssrc, -1);
			}
		    } else {
#endif
			ring = TRUE;
#ifdef RTP_SUPPORT
		    }
#endif
		    break;

                case 'S':             /* -Sn  --  Squelch at level n */
		    if (strlen(op + 1) == 0) {
			squelch = 50; /* Default squelch */
		    } else {
			squelch = atoi(op + 1);
		    }
		    break;

                case 'T':             /* -T -- Telephone (GSM) compression */
		    gsmcompress = TRUE;
		    lpccompress = adpcmcompress = FALSE;
		    break;

                case 'U':             /* -U  --  Print usage information */
                case '?':             /* -?  --  Print usage information */
		    usage();
		    return 0;

                case 'Z':             /* -Z"user1 user2..."  --  Send PGP
						 encrypted session key to
						 named users */
		    if (op[1] == 0) {
			curpgpkey[0] = FALSE;
		    } else {
			char c[80], f[40];
			FILE *kfile;
			FILE *pipe;
			long flen;

                        sprintf(f, "/tmp/.SF_SKEY%d", getpid());
                        sprintf(c, "pgp -fe +nomanual +verbose=0 %s >%s", op + 1, f);
#ifdef PGP_DEBUG
                        fprintf(stderr, "Encoding session key with: %s\n", c);
#endif
                        pipe = popen(c, "w");
			if (pipe == NULL) {
                            fprintf(stderr, "Unable to open pipe to: %s\n", c);
			    return 2;
			} else {
			    makeSessionKey(curpgpkey + 1, NULL, FALSE);
#ifdef PGP_DEBUG
			    {	
				int i;

                                fprintf(stderr, "Session key:", hostname);
				for (i = 0; i < 16; i++) {
                                    fprintf(stderr, " %02X", curpgpkey[i + 1] & 0xFF);
				}
                                fprintf(stderr, "\n");
			    }
#endif
			    /* The reason we start things off right with
                               "Special K" is to prevent the session key
			       (which can be any binary value) from
                               triggering PGP's detection of the file as
			       one already processed by PGP.  This causes an
			       embarrassing question when we are attempting
			       to run silent. */
                            curpgpkey[0] = 'K';
			    fwrite(curpgpkey, 17, 1, pipe);
			    curpgpkey[0] = FALSE;
			    fflush(pipe);
			    pclose(pipe);
			}
                        kfile = fopen(f, "r");
			if (kfile == NULL) {
                            fprintf(stderr, "Cannot open key file %s\n", f);
			} else {
			    fseek(kfile, 0L, 2);
			    flen = ftell(kfile);
			    rewind(kfile);
#ifdef PGP_DEBUG
                            fprintf(stderr, "PGP key buffer length: %d\n", flen);
#endif
#ifdef TINY_PACKETS
			    if (flen > (TINY_PACKETS -
				    (sizeof(soundbuf) - BUFL))) {
                                fprintf(stderr, "Warning: PGP key message exceeds %d packet size.\n", TINY_PACKETS);
			    }
#endif
			    if (pgpsb != NULL) {
				free(pgpsb);
			    }
			    pgpsb = (soundbuf *) malloc(((unsigned) flen) +
				      (TINY_PACKETS - (sizeof(soundbuf) - BUFL)));
			    if (pgpsb == NULL) {
                                fprintf(stderr, "Cannot allocate PGP sound buffer.\n");
				fclose(kfile);
				unlink(f);
				return 2;
			    }
			    fread(pgpsb->buffer.buffer_val, (int) flen, 1, kfile);
			    pgpsbl = flen;
			    pgpsb->buffer.buffer_len = htonl(flen);
			    pgpsb->compression = htonl(fKeyPGP);
#ifdef SENDMD5
			    MD5Init(&md5c);
			    MD5Update(&md5c, pgpsb->buffer.buffer_val, pgpsb->buffer.buffer_len);
			    MD5Final(pgpsb->sendinghost, &md5c);
#else
			    strcpy(pgpsb->sendinghost, hostname);
#endif
			    fclose(kfile);
			    curpgpkey[0] = TRUE;
			}
			unlink(f);
		    }
		    break;
	    }
	} else {
	    if (dests == NULL) {
		if (!addest(op)) {
		    return 1;
		}
	    } else {
		int ok = sendfile(op);
		if (ok != 0)
		    return ok;
		sentfile++;
	    }
	}
    }

    if (dests == NULL) {
	usage();
    } else {
	if (sentfile == 0) {
	    return sendfile(NULL);
	}
    }

    exiting();
    gsm_destroy(gsmh);
    desdone();
    return 0;
}
