/*

			Speak Freely for Unix
			   Echo Back Server

     Designed and implemented in January of 1996 by John Walker.

*/

#include "speakfree.h"

#define Echo_Base   0		      /* Echo server offset from speaker port */

#define Echo_Time   10		      /* Echo retransmit time in seconds */

static int debugforce = 0;	      /* Debugging forced / prevented ? */
static int whichport = Internet_Port + Echo_Base; /* Port to listen on (base address) */
static int sock;		      /* Input socket */
static struct sockaddr_in from;       /* Sending host address */
static struct sockaddr_in name;       /* Address of destination host */
static int fromlen;		      /* Length of sending host address */
static int showhosts = FALSE;	      /* Show host names that connect */
static int hosttimeout = 180 * 1000000L; /* Consider host idle after this time */
static char *prog;		      /* Program name */ 
static long timebase;		      /* Time in seconds at start of program */

static struct sockaddr_in lookhost;   /* Look who's listening host, if any */
static char *sdes = NULL;	      /* RTP SDES packet */
static int sdesl;		      /* RTP SDES packet length */
static int lwlport = Internet_Port + 2; /* Look Who's Listening port */
static unsigned long ssrc;	      /* RTP synchronisation source identifier */
static int lwltimer;		      /* Seconds before next LWL retransmit */
static int actives = 0; 	      /* Currently active hosts */

#define LWL_RETRANSMIT	(10 * 60)     /* Seconds between LWL updates */

#ifdef ECHO_FACE
static FILE *facefile = NULL;         /* User's face image */
#endif

struct connection *conn = NULL;       /* Chain of current connections */

#define Debug	    (debugforce != 0) /* Generate debug output */

#define TickTock    (10 * 1000000L)   /* Alarm interval in microseconds */
#define TockTock    (60 * 1000000L)   /* Alarm interval when no connections open */

#define AudioRelease (2 * TickTock)   /* Release audio when idle this long */

static long timerStep = TockTock;     /* Current timer step */

struct queuedPacket {
    struct queuedPacket *next;	      /* Next packet in chain */
    double when;		      /* When packet should be sent */
    struct sockaddr_in where;	      /* Where packet should be sent */
    int pktlen; 		      /* Packet length */
    char pktdata[2];		      /* Packet data */
};

struct queuedPacket *qph = NULL,      /* Queue of packets waiting to be echoed */
		    *qptail = NULL;
static int crit = FALSE;	      /* Queue critical section lock */
static int clash = FALSE;	      /* Critical section clash retry flag */

/*  ETIME  --  Edit time and date for log messages.  */

static char *etime()
{
    struct tm *t;
    time_t clock;
    static char s[20];

    time(&clock);
    t = localtime(&clock);
    sprintf(s, "%02d-%02d %02d:%02d", t->tm_mon + 1, t->tm_mday,
	       t->tm_hour, t->tm_min);
    return s;
}

/*  COMPRESSIONTYPE  --  Return a string describing the type of
			 compression employed in this buffer.  Assumes
			 this *is*, in fact, a sound buffer and not
			 an RTP packet, face data, etc.  */

static char *compressionType(msg)
  soundbuf *msg;
{
    return ((msg->compression & (fComp2X | fCompGSM)) ==
				(fComp2X | fCompGSM)) ?
                                "GSM+2X compressed" :
	   ((msg->compression & (fComp2X | fCompADPCM)) ==
				(fComp2X | fCompADPCM)) ?
                                "ADPCM+2X compressed" :
	   ((msg->compression & (fComp2X | fCompLPC)) ==
				(fComp2X | fCompLPC)) ?
                                "LPC+2X compressed" :
           ((msg->compression & fCompADPCM) ? "ADPCM compressed" :
           ((msg->compression & fCompLPC) ? "LPC compressed" :
           ((msg->compression & fComp2X) ? "2X compressed" :
           ((msg->compression & fCompGSM) ? "GSM compressed" :  "uncompressed"))));
}

/*  MAKESESSIONKEY  --	Generate session key.  */

static void makeSessionKey(key)
  char *key;
{
    struct MD5Context md5c;
    char s[1024];

    s[0] = 0;
    sprintf(s + strlen(s), "%u", getpid());
    sprintf(s + strlen(s), "%u", getppid());
    V getcwd(s + strlen(s), 256);
    sprintf(s + strlen(s), "%u", clock());
    V cuserid(s + strlen(s));
    sprintf(s + strlen(s), "%u", time(NULL));
#ifdef Solaris
    sysinfo(SI_HW_SERIAL, s + strlen(s), 12);
#else
    sprintf(s + strlen(s), "%u", gethostid());
#endif
    getdomainname(s + strlen(s), 256);
    gethostname(s + strlen(s), 256);
    sprintf(s + strlen(s), "%u", getuid());
    sprintf(s + strlen(s), "%u", getgid());
    MD5Init(&md5c);
    MD5Update(&md5c, s, strlen(s));
    MD5Final(key, &md5c);
}

/*  ULARM  --  Wrapper for setitimer() that looks like alarm()
	       but accepts a time in microseconds.  */

static void ularm(t)
  long t;
{
    struct itimerval it;

    it.it_value.tv_sec = t / 1000000L;
    it.it_value.tv_usec = t % 1000000L;
    it.it_interval.tv_sec = it.it_interval.tv_usec = 0;
    setitimer(ITIMER_REAL, &it, NULL);
}

/*  WINDTIMER  --  Reset alarm timer to appropriate value
                   depending on what's going on at the moment.  */

static void windtimer()
{
    if (actives > 0) {
	timerStep = TickTock;
    } else {
	timerStep = TockTock;
    }
    ularm(timerStep);
}

/*  SENDLWLMESSAGE  --	If enabled, send a message identifying us
                        to the selected Look Who's Listening server.  */

static void sendLwlMessage(dobye)
  int dobye;
{
    int sock;

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        perror("opening look who's listening socket");
	sdes = NULL;
	return;
    }

    if (connect(sock, (struct sockaddr *) &(lookhost), sizeof lookhost) >= 0) {
	if (dobye) {
	    char v[1024];
	    int l;

            l = rtp_make_bye(v, ssrc, "Exiting sfspeaker");
	    if (send(sock, v, l, 0) < 0) {
                perror("sending look who's listening BYE packet");
	    }
	} else {
	    if (send(sock, (char *) sdes, sdesl, 0) < 0) {
                perror("sending look who's listening source ID message");
	    }
	}
    } else {
        perror("connecting look who's listening socket");
    }
    close(sock);
}

/*  RELEASE  --  Signal-catching function which releases the audio
                 device if we haven't received anything to play in
		 the last minute. */

static void release()
{
    struct connection *c, *l, *n;

    if (crit) {
	clash = TRUE;
	ularm(60000000L);
    } else {
	struct timeval tp;
	struct timezone tzp;
	struct queuedPacket *qp;
	double td;

	gettimeofday(&tp, &tzp);
	td = (tp.tv_sec - timebase) * 1000000.0 + tp.tv_usec;

	/* Transmit any queued packets scheduled to go out within
	   epsilon of the current time. */

#define Timer_Epsilon	4000	      /* Timer epsilon in microseconds */
#define Timer_Latency	2000	      /* Dispatch latency adjustment */

#if (Timer_Latency >= Timer_Epsilon)
        error = "Timer epsilon must be greater than latency.";
#endif

	while (qph != NULL && ((qph->when - td) < Timer_Epsilon)) {
	    if (sendto(sock, qph->pktdata, qph->pktlen,
		0, (struct sockaddr *) &(qph->where),
		sizeof(struct sockaddr_in)) < 0) {
                perror("sending datagram message");
	    }
	    qp = qph;
	    qph = qp->next;
	    if (qph == NULL) {
		qptail = NULL;
	    }
	    free(qp);
	}

        /* Mark idle any hosts that haven't sent us anything recently. */

	c = conn;
	l = NULL;
	actives = 0;
	while (c != NULL) {
	    if ((c->con_timeout >= 0) && (c->con_timeout < hosttimeout)) {
		c->con_timeout += timerStep;
		actives++;
	    }
	    n = c->con_next;
	    if (c->con_timeout >= hosttimeout) {
		actives--;		  /* Oops--just went inactive */
		c->con_timeout = -1;	  /* Mark inactive */
		if (showhosts) {
                    fprintf(stderr, "%s: %s %s idle\n", prog, etime(), c->con_hostname);
		}
		if (l == NULL) {
		    conn = n;
		} else {
		    l->con_next = n;
		}
		free(c);
	    } else {
		l = c;
	    }
	    c = n;
	}

        /* Update our Look Who's Listening information if the
	   timeout has expired. */

	if (sdes != NULL && (tp.tv_sec - timebase) >= lwltimer) {
	    sendLwlMessage(FALSE);
	    lwltimer = (tp.tv_sec - timebase) + LWL_RETRANSMIT;
	}

	/* If we still own the sound device or have a connection
	   open, reset the timer. */

	if (qph != NULL) {
	    timerStep = (long) ((qph->when - td) - Timer_Latency);
	} else {
	    if (actives > 0) {
		timerStep = TickTock;
	    } else {
		timerStep = TockTock;
	    }
	}
	ularm(timerStep);
	if (Debug) {
            fprintf(stderr, "Tick: %.2f...\n", timerStep / 1000000.0);
	}
    }
    signal(SIGALRM, release);	      /* Set signal to handle timeout */
}

/*  EXITING  --  Catch as many program termination signals as
		 possible and clean up before exit.  */

static void exiting()
{

    if (sdes) {
	sendLwlMessage(TRUE);
    }
    exit(0);
}

/*  USAGE  --  Print how-to-call information.  */

static void usage()
{
    V fprintf(stderr, "%s  --  Speak Freely echo server.\n", prog);
    V fprintf(stderr, "            %s.\n", Relno);
    V fprintf(stderr, "\n");
    V fprintf(stderr, "Usage: %s [options]\n", prog);
    V fprintf(stderr, "Options:\n");
    V fprintf(stderr, "           -D               Enable debug output\n");
    V fprintf(stderr, "           -Pport           Listen on given port\n");
    V fprintf(stderr, "           -U               Print this message\n");
    V fprintf(stderr, "           -Vtimeout        Show hostnames that connect\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "by John Walker\n");
    V fprintf(stderr, "   E-mail: kelvin@fourmilab.ch\n");
    V fprintf(stderr, "   WWW:    http://www.fourmilab.ch/\n");
}

/*  Main program.  */

main(argc, argv)
  int argc;
  char *argv[];
{
    int i, length;
    struct soundbuf sb;
    struct connection *c;
    char *cp;
    int newconn;
    struct auhdr {		      /* .au file header */
	char magic[4];
	long hsize, dsize, emode, rate, nchan;
    };
    struct timeval tp;
    struct timezone tzp;
    struct queuedPacket *qp;

    prog = argv[0];
    gettimeofday(&tp, &tzp);
    timebase = tp.tv_sec;

    /* First pass option processing.  We have to first scan
       the options to handle any which affect creation of the
       socket.	One the second pass we can assume the socket
       already exists, allowing us to join multicast groups,
       etc. */

    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

                case 'P':             /* -Pport  --  Port to listen on */
		    whichport = atoi(op + 1);
		    break;

                case 'U':             /* -U  --  Print usage information */
                case '?':             /* -?  --  Print usage information */
		    usage();
		    return 0;
	    }
	} else {
	    usage();
	    return 2;
	}
    }

    /* Create the socket from which to read */

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0) {
        perror("opening datagram socket");
	return 1;
    }

    /* Create name with wildcards. */

    name.sin_family = AF_INET;
    name.sin_addr.s_addr = INADDR_ANY;
    name.sin_port = htons(whichport);
    if (bind(sock, (struct sockaddr *) &name, sizeof name) < 0) {
        perror("binding datagram socket");
	return 1;
    }


    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

                case 'D':             /* -D  --  Force debug output */
		    debugforce = 1;
		    break;

                case 'V':             /* -V  --  Show hostnames that connect */
		    showhosts = TRUE;
		    if (op[1] != 0) {
			int t = atoi(op + 1) * 1000000L;

			if (t > 0) {
			    if (t < (TickTock + 1)) {
				t = TickTock + 1;
			    }
			    hosttimeout = (t / TickTock) * TickTock;
			}
		    }
		    break;

	    }
	}
    }

    /* Find assigned port value and print it. */

    length = sizeof(name);
    if (getsockname(sock, (struct sockaddr *) &name, &length) < 0) {
        perror("getting socket name");
	return 1;
    }
#ifdef SHOW_SOCKET
    fprintf(stderr, "%s: socket port #%d\n", prog, ntohs(name.sin_port));
#endif

    /* Contact look who's listening host, if requested. */

    cp = getenv("SPEAKFREE_LWL_TELL");
    if (cp != NULL) {
	struct hostent *h;
	char md5key[16];
	char *ep;

        if ((ep = strchr(cp, ':')) != NULL) {
	    *ep = 0;
	    lwlport = atoi(ep + 1);
	}
	h = gethostbyname(cp);
	if (h != NULL) {
	    bcopy((char *) (h->h_addr), (char *) &lookhost.sin_addr.s_addr,
		sizeof lookhost.sin_addr.s_addr);
	    lookhost.sin_family = AF_INET;
	    lookhost.sin_port = htons(lwlport);

	    makeSessionKey(md5key);
	    bcopy(md5key, (char *) &ssrc, sizeof ssrc);

	    sdesl = rtp_make_sdes(&sdes, ssrc, whichport);
	    sendLwlMessage(FALSE);
	    lwltimer = LWL_RETRANSMIT;
	} else {
            fprintf(stderr, "%s: warning, SPEAKFREE_LWL_TELL host %s unknown.\n",
		prog, cp);
	}
    }

#ifdef ECHO_FACE

    /*	See if the user has specified a face file.  If so,
	try to open it.  */

    {   char *cp = getenv("SPEAKFREE_FACE");

	if (cp != NULL) {
            if ((facefile = fopen(cp, "r")) == NULL) {
                fprintf(stderr, "%s: cannot open SPEAKFREE_FACE file %s\n",
		    prog, cp);
	    }
	}
    }
#endif

    signal(SIGHUP, exiting);	      /* Set signal to handle termination */
    signal(SIGINT, exiting);	      /* Set signal to handle termination */
    signal(SIGTERM, exiting);	      /* Set signal to handle termination */
    signal(SIGALRM, release);	      /* Set signal to handle timeout */
    windtimer();		      /* Set periodic status update  */

    /* Read from the socket. */

    while (TRUE) {
	int rll;

	fromlen = sizeof(from);
	if ((rll = recvfrom(sock, (char *) &sb, sizeof sb, 0, (struct sockaddr *) &from, &fromlen)) < 0) {
#ifdef sgi
	    if (errno != EINTR) {
                perror("receiving datagram packet");
	    }
#endif
	    continue;
	}
	gettimeofday(&tp, &tzp);      /* Get time packet received */
#ifdef SHOW_SOCKET
	if (Debug) {
            fprintf(stderr, "%s: %d bytes read from socket.\n", prog, rll);
	}
#endif

	/* See if this connection is active.  If not, initialise a new
	   connection. */

	newconn = FALSE;
	c = conn;
	while (c != NULL) {
	    if (memcmp(&from.sin_addr, &(c->con_addr),
		       sizeof(struct in_addr)) == 0) {
		break;
	    }
	    c = c->con_next;
	}
	if (c == NULL) {
	    c = (struct connection *) malloc(sizeof(struct connection));
	    if (c != NULL) {
		struct hostent *h;

		newconn = TRUE;
		c->con_next = conn;
		c->pgpkey[0] = FALSE;
		bzero(c->keymd5, 16);
		conn = c;
		bcopy(&from.sin_addr, &(c->con_addr),
		    sizeof(struct in_addr));
		h = gethostbyaddr((char *) &from.sin_addr, sizeof(struct in_addr),
				  AF_INET);
		if (h == NULL) {
		    strcpy(c->con_hostname, inet_ntoa(from.sin_addr));
		} else {
		    strcpy(c->con_hostname, h->h_name);
		}
	    }
	} else if (c->con_timeout == -1) {
	    newconn = TRUE;
	}

	/* Initialise fields in connection.  Only fields which need to
	   be reinitialised when a previously idle host resumes activity
	   need be set here. */

	if (newconn) {
	    c->face_file = NULL;
	    c->face_filename[0] = 0;
	    c->face_viewer = 0;
	    c->face_stat = FSinit;
	    c->face_address = 0L;
	    c->face_retry = 0;
	    c->con_compmodes = -1;
	}

	if (c != NULL) {
	    /* Reset connection timeout. */
	    c->con_timeout = 0;
	    if (newconn) {
		if (showhosts) {
                    fprintf(stderr, "%s: %s %s connect\n", prog, etime(), c->con_hostname);
		}
	    }
	} else {
	    continue;
	}

#ifdef RTP_SUPPORT

	/* If this is an RTP packet, transmogrify it into a sound
	   buffer we can understand. */

	if (isrtp((unsigned char *) &sb, rll, c)) {
	    if (sb.buffer.buffer_len == 0) {
                fprintf(stderr, "Ignoring unparseable RTP packet.\n");
		continue;
	    }
	} else {
#endif

	    /* Convert relevant fields from network to host
	       byte order, if necessary. */

	    sb.compression = ntohl(sb.compression);
	    sb.buffer.buffer_len = ntohl(sb.buffer.buffer_len);
#ifdef RTP_SUPPORT
	}
#endif

	/* If this is a face request and we have a face file open,
	   respond to it.  Note that servicing of face file data requests
	   is stateless. */

	if (sb.compression & fFaceData) {
	    if (sb.compression & faceRequest) {
		long l;

		/* Request for face data. */

#ifdef ECHO_FACE
		if (facefile != NULL) {
		    fseek(facefile, sb.buffer.buffer_len, 0);
		    *((long *) sb.buffer.buffer_val) = htonl(sb.buffer.buffer_len);
		    l = fread(sb.buffer.buffer_val + sizeof(long),
			1, 512 - (sizeof(long) + (sizeof(soundbuf) - BUFL)), facefile);
		    sb.compression = fFaceData | faceReply;
		    if (Debug) {
                        fprintf(stderr, "%s: sending %d bytes of face data at %ld to %s\n",
			    prog, l, ntohl(*((long *) sb.buffer.buffer_val)), c->con_hostname);
		    }
		    l += sizeof(long);
		} else {
#endif
		    /* No face file.  Shut down requestor. */
		    sb.compression = fFaceData | faceLess;
		    l = 0;
#ifdef ECHO_FACE
		}
#endif
		bcopy((char *) &(from.sin_addr), (char *) &(name.sin_addr),
		    sizeof(struct in_addr));
		name.sin_port = htons(Internet_Port);

		sb.compression = htonl(sb.compression);
		sb.buffer.buffer_len = htonl(l);
		if (sendto(sock, (char *) &sb,
		    (int) ((sizeof(struct soundbuf) - BUFL) + l),
		    0, (struct sockaddr *) &(name), sizeof name) < 0) {
                    perror("sending face image data");
		}
	    }
	    continue;		      /* Done with packet */
	}

	/* Add the packet to the queue of packets awaiting
	   retransmission. */

	qp = (struct queuedPacket *) malloc(sizeof(struct queuedPacket) + rll);
	if (qp != NULL) {
	    qp->next = NULL;
	    qp->when = ((tp.tv_sec + Echo_Time) - timebase) * 1000000.0 + tp.tv_usec;
	    bcopy((char *) &from, (char *) &(qp->where),
		sizeof(struct sockaddr_in));
	    qp->where.sin_port = htons(Internet_Port);
	    sb.compression &= ~fLoopBack;    /* Prevent infinite loopback */
#ifdef ECHO_FACE
	    if (facefile != NULL) {
		sb.compression |= fFaceOffer;
	    }
#endif
	    sb.compression = htonl(sb.compression);
	    sb.buffer.buffer_len = htonl(sb.buffer.buffer_len);
	    qp->pktlen = rll;
	    bcopy((char *) &sb, qp->pktdata, rll);
	    sb.compression = ntohl(sb.compression);
	    sb.buffer.buffer_len = ntohl(sb.buffer.buffer_len);
	    crit = TRUE;
	    if (qptail == NULL) {
		qph = qptail = qp;
		ularm(Echo_Time * 1000000L);
	    } else {
		qptail->next = qp;
		qptail = qp;
	    }
	    crit = FALSE;
	    if (clash) {
		release();
	    }
	}

	if (showhosts && (c->con_compmodes != (fCompressionModes &
		sb.compression))) {
	    c->con_compmodes = fCompressionModes & sb.compression;
	    if (showhosts) {
                fprintf(stderr, "%s: %s sending %s.\n", prog, c->con_hostname,
			compressionType(&sb));
	    }
	}
	if (Debug) {
            fprintf(stderr, "%s: echoing %d %s bytes from %s.\n",
		    prog, sb.buffer.buffer_len,
		    compressionType(&sb), c->con_hostname);
	}
    }
#ifdef MEANS_OF_EXIT
    close(sock);
    exiting();
#ifdef ECHO_FACE
    if (facefile != NULL) {
	fclose(facefile);
    }
#endif
    return 0;
#endif
}
