/*

			Speak Freely for Unix
                     Look Who's Listening Server

*/

#include "speakfree.h"

#define TockTock    120 	      /* Timeout check frequency, seconds */
#define TimeoutTime 15 * 60	      /* No-response timeout interval, seconds */

#define MaxReplyPacket 400	      /* Maximum length of reply data */

static int sock;		      /* Our socket */
static char *prog;		      /* Program name */ 

static int lwlport = Internet_Port + 2; /* Look Who's Listening port */
static int debugging = FALSE;	      /* Debug mode enabled */
static int verbose = FALSE;	      /* Show connections/disconnects */
static int prolix = FALSE;	      /* Extremely verbose (show queries) */
#ifdef HEXDUMP
static int hexdump = FALSE;	      /* Dump received packets in hex ? */
#endif
static char *htmlFile = NULL;	      /* HTML file name base */
static char *htmlPrivateFile = NULL;  /* HTML private directory file name base */
static int htmlTime = 1 * 60;	      /* HTML file update time */
static time_t htmlLast = 0;	      /* HTML last update time */
static int htmlChange = TRUE;	      /* Change since last HTML update ? */
static char *message = NULL;	      /* Server information message */
static int messagel;		      /* Length of server information message */

struct lwl {
    struct lwl *next;		      /* Next connection */
    time_t ltime;		      /* Time of last update */
    unsigned long ssrc; 	      /* Session source descriptor */
    long naddr; 		      /* Internet address */
    short port; 		      /* Port address */
    char *cname;		      /* Canonical name */ 
    char *name; 		      /* User name */
    char *email;		      /* Electronic mail address */
    char *phone;		      /* Telephone number */
    char *loc;			      /* Geographic location */
    char *tool; 		      /* Application name */
};

static struct lwl *conn = NULL;       /* Chain of current connections */

/*  ETIME  --  Edit time and date for log messages.  */

static char *etime(gmt)
  int gmt;
{
    struct tm *t;
    time_t clock;
    static char s[20];

    time(&clock);
    if (gmt) {
	t = gmtime(&clock);
    } else {
	t = localtime(&clock);
    }
    sprintf(s, "%02d-%02d %02d:%02d", t->tm_mon + 1, t->tm_mday,
	       t->tm_hour, t->tm_min);
    return s;
}

/*  ESTIME  --	Edit short time.  */

static char *estime()
{
    struct tm *t;
    time_t clock;
    static char s[20];

    time(&clock);
    t = localtime(&clock);
    sprintf(s, "%02d:%02d", t->tm_hour, t->tm_min);
    return s;
}

/*  DUPSDESITEM  --  Make a copy of an SDES item and advance the pointer
		     past it.  */

static char *dupSdesItem(cp)
  char **cp;
{
    char *ip = *cp, *bp;
    int l = ip[1] & 0xFF;

    bp = malloc(l + 1);
    if (bp != NULL) {
	bcopy(ip + 2, bp, l);
	bp[l] = 0;
    }
    *cp = ip + l + 2;
    return bp;
}

/*  DESTROYLWL	--  Release storage associated with an LWL list item.  */

static void destroyLwl(lw)
  struct lwl *lw;
{
    if (lw->cname != NULL) {
	free(lw->cname);
    }
    if (lw->name != NULL) {
	free(lw->name);
    }
    if (lw->email != NULL) {
	free(lw->email);
    }
    if (lw->phone != NULL) {
	free(lw->phone);
    }
    if (lw->loc != NULL) {
	free(lw->loc);
    }
    if (lw->tool != NULL) {
	free(lw->tool);
    }
    free(lw);
}

/*  DUMPLWL  --  Dump an LWL on the specified stream.  */

static void dumpLwl(fo, lw)
  FILE *fo;
  struct lwl *lw;
{
    struct sockaddr_in s;

    s.sin_addr.s_addr = lw->naddr;
    fprintf(fo, "SSRC = %X IP = %s Port %d %s", lw->ssrc, inet_ntoa(s.sin_addr),
		lw->port, ctime(&(lw->ltime)));
    if (lw->cname != NULL) {
        fprintf(fo, "    CNAME = %s\n", lw->cname);
    }
    if (lw->name != NULL) {
        fprintf(fo, "     NAME = %s\n", lw->name);
    }
    if (lw->email != NULL) {
        fprintf(fo, "    EMAIL = %s\n", lw->email);
    }
    if (lw->phone != NULL) {
        fprintf(fo, "    PHONE = %s\n", lw->phone);
    }
    if (lw->loc != NULL) {
        fprintf(fo, "      LOC = %s\n", lw->loc);
    }
    if (lw->tool != NULL) {
        fprintf(fo, "     TOOL = %s\n", lw->tool);
    }
}

/*  DUMPLWLCHAIN  --  Dump all current connections.  */

static void dumpLwlChain(fo)
  FILE *fo;
{
    struct lwl *lw = conn;

    fprintf(fo, "\n==========  %s  ==========\n", etime(FALSE));
    while (lw != NULL) {
        fprintf(fo, "\n");
	dumpLwl(fo, lw);
	lw = lw->next;
    }
}

/*  LOGLWL  --	Generate log entry for LWL event.  */

static void logLwl(lw, event)
  struct lwl *lw;
  char *event;
{
    char ipport[40], deef[256];
    struct sockaddr_in u;

    u.sin_addr.s_addr = lw->naddr;
    sprintf(ipport, "%s:%d", inet_ntoa(u.sin_addr), lw->port);
    sprintf(deef, "%s %s", ipport, lw->email ? lw->email : lw->cname);
    if (lw->name) {
        sprintf(deef + strlen(deef), " (%s)", lw->name);
    }
    printf("%s %s%s\n", estime(), event, deef);
}

/*  MAKEHTML  --  Create an HTML file showing current connections.  If
                  "private" is set, exact-match names are included
		  in the HTML file; this allows creation of a non-exported
                  active site file for the system manager's use.  */

static int makeHTML(fname, private)
  char *fname;
  int private;
{
    FILE *of;
    char f[132], fn[132];

    strcpy(f, fname);
    strcat(f, ".new");
    of = fopen(f, "w");
    if (of != NULL) {
	struct lwl *lw = conn;

#define P(x) fprintf(of, x)

        P("<html>\n<head>\n<title>\nSpeak Freely: Active Sites\n");
        P("</title>\n</head>\n\n<body>\n<center>\n<h1>");
        P("Speak Freely: Active Sites</h1>\n<h2>");
        fprintf(of, "As of %s UTC</h2>\n</center>\n<p>\n", etime(TRUE));

	if (lw == NULL) {
            P("<h2>No sites active.</h2>\n");
	} else {
	    int i = 0;

	    while (lw != NULL) {
                if (private || ((lw->email == NULL || lw->email[0] != '*') &&
                    (lw->cname[0] != '*'))) {
		    i++;
		}
		lw = lw->next;
	    }
	    lw = conn;
	    if (i == 0) {
		/* Fib about number of sites active if all active sites
                   don't want a directory listing. */
                P("<h2>No sites active.</h2>\n");
	    } else {
                fprintf(of, "<h2>%d site%s active.</h2>\n",
                    i, i == 1 ? "" : "s");
                P("<pre>\n");
		while (lw != NULL) {
                    if (private || ((lw->email == NULL || lw->email[0] != '*') &&
                        (lw->cname[0] != '*'))) {
			char ipport[40];
			struct tm *lt;
			struct sockaddr_in u;

			lt = localtime(&lw->ltime);
			u.sin_addr.s_addr = lw->naddr;
                        sprintf(ipport, "%s:%d", inet_ntoa(u.sin_addr), lw->port);
                        fprintf(of, "\n%-24s %-48s %02d:%02d\n", ipport, lw->cname,
			    lt->tm_hour, lt->tm_min);
			if (lw->name != NULL) {
                            fprintf(of, "%25s%s\n", "", lw->name);
			}
			if (lw->loc != NULL) {
                            fprintf(of, "%25s%s\n", "", lw->loc);
			}
			if (lw->phone != NULL) {
                            fprintf(of, "%25sPhone:  %s\n", "", lw->phone);
			}
			if (lw->email != NULL) {
                            fprintf(of, "%25sE-mail: %s\n", "", lw->email);
			}
		    }
		    lw = lw->next;
		}
                P("</pre>\n");
	    }
	}

        P("</body>\n</html>\n");
	fclose(of);
	strcpy(fn, fname);
        strcat(fn, ".html");
	rename(f, fn);
	if (debugging) {
            fprintf(stderr, "%s: updated %s\n", prog, fn);
	}
	return TRUE;
    }
    return FALSE;
}

/*  UPDHTML  --  Update HTML if necessary.  */

static void updHTML()
{
    time_t now;

    if (((htmlFile != NULL) || (htmlPrivateFile != NULL)) && htmlChange && 
	((htmlTime <= 0) || (((now = time(NULL)) - htmlLast) > htmlTime))) {
	if (htmlFile != NULL) {
	    makeHTML(htmlFile, FALSE);
	}
	if (htmlPrivateFile != NULL) {
	    makeHTML(htmlPrivateFile, TRUE);
	}
	htmlLast = now;
	htmlChange = FALSE;
    }
}

/*  CHANGED  --  Indicate a change which may require updating
		 the HTML file.  */

static void changed()
{
    htmlChange = TRUE;
    updHTML();
}

/*  LCASE  --  Convert a string to lower case.	*/

static void lcase(s)
  char *s;
{
    while (*s) {
	if (isupper(*s)) {
	    *s = tolower(*s);
	}
	s++;
    }
}

/*  QUERYMATCH	--  Determing if query matches a given item.  */

static int queryMatch(q, l)
  char *q;
  struct lwl *l;
{
    char ts[1024];
    int exact = FALSE;
    char *s;

    s = ts;

    /* We give preference to an E-mail address, if given, to a
       canonical address since it's the user's best known published
       identity on the net.  This, of course, runs the risk of
       spoofing, but since anybody can send us an RTCP packet with
       whatever cname they like, there's no added security in insisting
       on using it. */

    strcpy(s, (l->email == NULL) ? l->cname : l->email);
    lcase(s);

    if (*q == '*') {
	exact = TRUE;
	q++;
    }
    if (*s == '*') {
	exact = TRUE;
	s++;
    }
    /* Even if we're using the E-mail name, allow an asterisk on the
       canonical name to require an exact match. */
    if (l->cname[0] == '*') {
	exact = TRUE;
    }

    if (exact) {
	return strcmp(s, q) == 0;
    }

    if (strstr(s, q)) {
	return TRUE;
    }
    if (l->name != NULL) {
	strcpy(s, l->name);
	lcase(s);
	if (strstr(s, q)) {
	   return TRUE;
	}
    }
    return FALSE;
}

/*  RELEASE  --  Check for connections who haven't send us an update
		 once in the timeout interval and close them.  They
		 probably went away without having the courtesy to
		 say good-bye.	*/

static void release()
{
    struct lwl *lw = conn, *llr = NULL, *lf;
    time_t now = time(NULL);

    while (lw != NULL) {
	if ((now - lw->ltime) > TimeoutTime) {
	    lf = lw;
	    if (llr == NULL) {
		conn = lw->next;
		lw = conn;
	    } else {
		llr->next = lw->next;
		lw = lw->next;
	    }
/*
fprintf(stderr, "\nTiming out:\n");
dumpLwl(stderr, lf);
*/
	    if (verbose) {
                logLwl(lf, "Timeout: ");
	    }
	    htmlChange = TRUE;
	    destroyLwl(lf);
	} else {
	    llr = lw;
	    lw = lw->next;
	}
    }
    updHTML();                        /* Update HTML if something's changed recently */
    signal(SIGALRM, release);	      /* Reset signal to handle timeout (Sys V) */
    alarm(TockTock);		      /* Reset the timer */
}

/*  PLUMBER  --  Catch SIGPIPE signal which occurs when remote user
		 disconnects before we try to send the reply.  */

static void plumber()
{
/*
fprintf(stderr, "Caught SIGPIPE--continuing.\n");
*/
    signal(SIGPIPE, plumber);	      /* Reset signal just in case */
}

/*  EXITING  --  Release socket in case of termination.  */

static void exiting()
{
/*
fprintf(stderr, "Exiting.\n");
*/
    shutdown(sock, 2);
    close(sock);
    exit(0);
}

/*  USAGE  --  Print how-to-call information.  */

static void usage()
{
    V fprintf(stderr, "%s  --  Speak Freely: Look Who's Listening Server.\n", prog);
    V fprintf(stderr, "               %s.\n", Relno);
    V fprintf(stderr, "\n");
    V fprintf(stderr, "Usage: %s [options]\n", prog);
    V fprintf(stderr, "Options:\n");
    V fprintf(stderr, "           -D         Debugging output on stderr\n");
    V fprintf(stderr, "           -Hpath     Write HTML status on base path\n");
    V fprintf(stderr, "           -Insec     Interval between HTML updates\n");
    V fprintf(stderr, "           -Mfile     Load server message from file\n");
    V fprintf(stderr, "           -Pport     Listen on given port\n");
    V fprintf(stderr, "           -U         Print this message\n");
    V fprintf(stderr, "           -V[V]      List connections and disconnects [-VV: all packets]\n");
#ifdef HEXDUMP
    V fprintf(stderr, "           -X         Dump packets in hex\n");
#endif
    V fprintf(stderr, "           -Zpath     Write HTML private directory on base path\n");
    V fprintf(stderr, "\n");
    V fprintf(stderr, "by John Walker\n");
    V fprintf(stderr, "   E-mail: kelvin@fourmilab.ch\n");
    V fprintf(stderr, "   WWW:    http://www.fourmilab.ch/\n");
}

/*  Main program.  */

main(argc, argv)
  int argc;
  char *argv[];
{
    int i, length;
    struct sockaddr_in name;

    /*	Process command line options.  */

    prog = argv[0];
    for (i = 1; i < argc; i++) {
	char *op, opt;

	op = argv[i];
        if (*op == '-') {
	    opt = *(++op);
	    if (islower(opt)) {
		opt = toupper(opt);
	    }

	    switch (opt) {

                case 'D':             /* -D  --  Debug output to stderr */
		    debugging = TRUE;
		    break;

                case 'H':             /* -Hname  --  HTML file name base path */
		    htmlFile = op + 1;
		    break;

                case 'I':             /* -Isec  --  Interval between HTML updates */
		    htmlTime = atoi(op + 1);
		    break;

                case 'M':             /* -Mfile  --  Load server message from file */
		    {
                        FILE *fp = fopen(op + 1, "r");
			long fl;

			if (fp == NULL) {
                            fprintf(stderr, "%s: can't open server message file %s\n", prog, op + 1);
			    return 2;
			}
			fseek(fp, 0L, 2);
			fl = ftell(fp);
			rewind(fp);
			messagel = ((int) fl) + 13;
			message = (char *) malloc(messagel);
			if (message != NULL) {
			    rtcp_t *rp = (rtcp_t *) message;

#ifdef RationalWorld
			    rp->common.version = RTP_VERSION;
			    rp->common.p = 0;
			    rp->common.count = 1;
			    rp->common.pt = RTCP_APP;
#else
			    *((short *) rp) = htons((RTP_VERSION << 14) |
						    RTCP_APP | (1 << 8));
#endif
			    rp->r.sdes.src = 0;
                            bcopy("SFmr", message + 8, 4);
			    fread(message + 12, (int) fl, 1, fp);
			    message[messagel - 1] = 0;
			}
			fclose(fp);
		    }
		    break;

                case 'P':             /* -Pport  --  Port to listen on */
		    lwlport = atoi(op + 1);
		    break;

                case 'U':             /* -U  --  Print usage information */
                case '?':             /* -?  --  Print usage information */
		    usage();
		    return 0;

                case 'V':             /*  -V  -- Show connects/disconnects */
		    verbose = TRUE;
                    if (op[1] == 'v' || op[1] == 'V') {
			prolix = TRUE;
		    }
		    break;

#ifdef HEXDUMP
                case 'X':             /* -X  --  Dump packets in hex */
		    hexdump = TRUE;
		    break;
#endif

                case 'Z':             /* -Zname  --  HTML private file name base path */
		    htmlPrivateFile = op + 1;
		    break;
	    }
	} else {
	    usage();
	    return 2;
	}
    }

    /* If no server message has been loaded, create a void one. */

    if (message == NULL) {
	rtcp_t *rp;

	messagel = 13;
	message = (char *) malloc(messagel);
	rp = (rtcp_t *) message;

#ifdef RationalWorld
	rp->common.version = RTP_VERSION;
	rp->common.p = 0;
	rp->common.count = 1;
	rp->common.pt = RTCP_APP;
#else
	*((short *) rp) = htons((RTP_VERSION << 14) | RTCP_APP | (1 << 8));
#endif
	rp->r.sdes.src = 0;
        bcopy("SFmr", message + 8, 4);
	message[messagel - 1] = 0;
    }


    /* Create the socket from which to read */

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        perror("opening stream socket");
	return 1;
    }

    /* Create name with wildcards. */

    name.sin_family = AF_INET;
    name.sin_addr.s_addr = INADDR_ANY;
    name.sin_port = htons(lwlport);
    if (bind(sock, (struct sockaddr *) &name, sizeof name) < 0) {
        perror("binding stream socket");
	return 1;
    }

    signal(SIGHUP, exiting);	      /* Set signal to handle termination */
    signal(SIGINT, exiting);	      /* Set signal to handle termination */
    signal(SIGTERM, exiting);	      /* Set signal to handle termination */

    signal(SIGPIPE, plumber);         /* Catch "broken pipe" signals from disconnects */

    if (listen(sock, 5) < 0) {
        perror("calling listen for socket");
    }

    /* Find assigned port value and print it. */

    length = sizeof(name);
    if (getsockname(sock, (struct sockaddr *) &name, &length) < 0) {
        perror("getting socket name");
	return 1;
    }
#ifdef SHOW_SOCKET
    fprintf(stderr, "%s: socket port #%d\n", prog, ntohs(name.sin_port));
#endif

    signal(SIGALRM, release);	      /* Set signal to handle timeout */
    alarm(TockTock);		      /* Set alarm clock to purge idle hosts */
    changed();			      /* Create initial HTML file */

    /* Process requests from the socket. */

    while (TRUE) {
	int csock, rll;
	char zp[1024];
	rtcp_t *rp;
	short srp;
	struct sockaddr_in from;      /* Sending host address */
	int fromlen;		      /* Length of sending host address */
	struct lwl *lw; 	      /* Parsed packet */
	char *p;

	errno = 0;
	do {
	    fromlen = sizeof from;
	    csock = accept(sock, (struct sockaddr *) &from, &fromlen);
	    if (csock >= 0) {
		break;
	    }
	} while (errno == EINTR);
	if (csock < 0) {
            perror("accepting connection to socket");
	    return 1;
	}
	if (prolix) {
            printf("%s: %s accept\n", prog, inet_ntoa(from.sin_addr));
	}

	rll = recv(csock, zp, sizeof zp, 0);
#ifdef HEXDUMP
	if (hexdump) {
            fprintf(stderr, "%s: %d bytes read from socket.\n", prog, rll);
	    xd(stderr, zp, rll, TRUE);
	}
#endif

	/* Walk through the individual items in a possibly composite
           packet until we locate an item we're interested in.  This
	   allows us to accept packets that comply with the RTP standard
	   that all RTCP packets begin with an SR or RR.  */

	p = zp;

	while ((p[0] >> 6 & 3) == RTP_VERSION) {
	    int pt = p[1] & 0xFF;

	    if (pt == RTCP_SDES || pt == RTCP_BYE || pt == RTCP_APP) {
		break;
	    }
	    /* If not of interest to us, skip to next subpacket. */
	    p += (ntohs(*((short *) (p + 2))) + 1) * 4;
	}

	/* Examine the packet to see what kind of request it is.
           Note that since all the packets we're interested in can
	   be parsed without knowing their actual length in bytes, we
	   can ignore the possible presence of padding. */

	rp = (rtcp_t *) p;
	srp = ntohs(*((short *) p));
	if (
#ifdef RationalWorld
	    rp->common.version == RTP_VERSION && /* Version ID correct */
	    rp->common.count == 1
#else
	    (((srp >> 14) & 3) == RTP_VERSION) &&
	    (((srp >> 8) & 0x1F) == 1)
#endif
	   ) {

	    switch (
#ifdef RationalWorld
		    rp->common.pt
#else
		    srp & 0xFF
#endif
		   ) {

		/*  SDES packet.  This is a notification by a listening
                                  that it's just started or is still
				  listening.  If the host was previously
				  active we replace the old parameters
				  with the new in case something has
				  changed.  The timeout is reset.  Identity
				  of host is by canonical name, not SSRC,
				  since the host may have restarted with
				  a new SSRC without having sent us a BYE.  */

		case RTCP_SDES:

		    /* Parse the fields out of the SDES packet. */

		    {
			char *cp = (char *) rp->r.sdes.item,
			     *lp = cp + (ntohs(rp->common.length) * 4);
			struct lwl *lr, *llr;

			lw = (struct lwl *) malloc(sizeof(struct lwl));
			if (lw != NULL) {
			    bzero((char *) lw, sizeof(struct lwl));

			    lw->ssrc = rp->r.sdes.src;
			    lw->naddr = from.sin_addr.s_addr;
			    lw->port = from.sin_port;
			    lw->ltime = time(NULL);
			    if (prolix) {
                                printf("%s: %s SDES %08X\n", prog,
				    inet_ntoa(from.sin_addr), lw->ssrc);
			    }
			    while (cp < lp) {
				switch ((*cp) & 0xFF) {

				    case RTCP_SDES_CNAME:
					lw->cname = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_NAME:
					lw->name = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_EMAIL:
					lw->email = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_PHONE:
					lw->phone = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_LOC:
					lw->loc = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_TOOL:
					lw->tool = dupSdesItem(&cp);
					break;

				    case RTCP_SDES_PRIV:
					{
					    char *zp = dupSdesItem(&cp);

					    if (zp != NULL) {
						if (zp[0] == 1 &&
                                                    zp[1] == 'P') {
						    lw->port = atoi(zp + 2);
						}
						free(zp);
					    }
					}
					break;

				    case RTCP_SDES_END:
					cp = lp;
					break;

				    default:
					{
					    char *zp = dupSdesItem(&cp);

					    if (zp != NULL) {
						free(zp);
					    }
					}
					break;
				}
			    }
			    if (debugging) {
				dumpLwl(stderr, lw);
			    }

			    /* Search chain and see if a user with this
			       name is already know.  If so, replace the
			       entry with this one. */

			    if (lw->cname != NULL) {
				lr = conn;
				llr = NULL;

				while (lr != NULL) {
				    char *p = lw->cname, *q = lr->cname;

                                    if (*p == '*') {
					p++;
				    }
                                    if (*q == '*') {
					q++;
				    }
				    if (strcmp(p, q) == 0) {
					lw->next = lr->next;
					if (llr == NULL) {
					    conn = lw;
					} else {
					    llr->next = lw;
					}
					destroyLwl(lr);
					lw = NULL;
					break;
				    }
				    llr = lr;
				    lr = lr->next;
				}

                                /* If we didn't find an entry already in the
				   chain, link in the new entry.  */

				if (lw != NULL) {
				    lw->next = conn;
				    conn = lw;
				    if (verbose) {
                                        logLwl(lw, "Connect: ");
				    }
				    changed();
				}
			    }
			}
		    }
		    break;

		/*  BYE packet.  This is sent when a listening host is
				 ceasing to listen in an orderly manner.
				 Identity here is by SSRC, since the
				 host is assumed to have properly announced
                                 itself.  Besides, that's the only ID in
				 the RTCP BYE packet, so it had darned
				 well be sufficient.  */

		case RTCP_BYE:
		    {
			struct lwl *lr = conn, *llr = NULL;

			if (prolix) {
                            printf("%s: %s BYE %08X\n", prog,
				inet_ntoa(from.sin_addr), rp->r.bye.src[0]);
			}
			while (lr != NULL) {
			    if (rp->r.bye.src[0] == lr->ssrc) {
				if (llr == NULL) {
				    conn = lr->next;
				} else {
				    llr->next = lr->next;
				}
				if (debugging) {
                                    fprintf(stderr, "Releasing:\n");
				    dumpLwl(stderr, lr);
				}
				if (verbose) {
                                    logLwl(lr, "Bye:     ");
				}
				changed();
				destroyLwl(lr);
				break;
			    }
			    llr = lr;
			    lr = lr->next;
			}
		    }
		    break;

		/*  Application request packets.  The following application
		    extensions implement the various queries a client can make
		    regarding the current state of the listening host list.  */

		case RTCP_APP:
		    if (prolix) {
                        printf("%s: %s APP %.4s\n", prog,
			    inet_ntoa(from.sin_addr), p + 8);
		    }

		    /*	SFlq  --  Pattern match in cname and name and
				  return matches, up to the maximum
				  packet size.	If either the query string
				  or the canonical name begins with an
				  asterisk, the asterisk(s) is(/are) ignored
				  and a precise match with the canonical
				  name is required.  */

                    if (bcmp(p + 8, "SFlq", 4) == 0) {
			struct lwl *lr = conn;
			char b[1500];
			rtcp_t *rp = (rtcp_t *) b;
			char *ap;
			int l, scandex = 0;

#ifdef RationalWorld
			rp->common.version = RTP_VERSION;
			rp->common.p = 0;
			rp->common.count = 0;
			rp->common.pt = RTCP_SDES;
#else
			*((short *) rp) = htons((RTP_VERSION << 14) |
						 RTCP_SDES | (0 << 8));
#endif

			ap = (char *) &(rp->r.sdes.src);

			if (prolix) {
                            printf("%s: %s query \"%s\"\n", prog, inet_ntoa(from.sin_addr), p + 12);
			}
			lcase(p + 12);
			while (lr != NULL) {
			    if (queryMatch(p + 12, lr)) {
				char s[20];

#define addSDES(item, text) *ap++ = item; *ap++ = l = strlen(text); \
			    bcopy(text, ap, l); ap += l

				*((unsigned long *) ap) = lr->ssrc;
				ap += sizeof(unsigned long);
				addSDES(RTCP_SDES_CNAME, lr->cname +
                                    (lr->cname[0] == '*' ? 1 : 0));

				if (lr->name != NULL) {
				    addSDES(RTCP_SDES_NAME, lr->name);
				}

				if (lr->email != NULL) {
				    addSDES(RTCP_SDES_EMAIL, lr->email +
                                      (lr->email[0] == '*' ? 1 : 0));
				}

				if (lr->phone != NULL) {
				    addSDES(RTCP_SDES_PHONE, lr->phone);
				}

				if (lr->loc != NULL) {
				    addSDES(RTCP_SDES_LOC, lr->loc);
				}

				if (lr->tool != NULL) {
				    addSDES(RTCP_SDES_TOOL, lr->tool);
				}

                                sprintf(s, "\001P%d", lr->port);
				addSDES(RTCP_SDES_PRIV, s);

				{
				    struct sockaddr_in u;

				    u.sin_addr.s_addr = lr->naddr;
                                    sprintf(s, "\001I%s", inet_ntoa(u.sin_addr));
				    addSDES(RTCP_SDES_PRIV, s);
				}

                                sprintf(s, "\001T%lu", lr->ltime);
				addSDES(RTCP_SDES_PRIV, s);

                                /* If we're over the packet size limit,
                                   let the user know there's more to be
				   retrieved starting at the given offset. */

				if ((ap - b) > MaxReplyPacket) {
                                    sprintf(s, "\001M%d", scandex);
				    addSDES(RTCP_SDES_PRIV, s);
				}
				*ap++ = RTCP_SDES_END;

				/* Pad to next 32 bit boundary. */

				while (((ap - b) & 3) != 0) {
				    *ap++ = RTCP_SDES_END;
				}
#ifdef RationalWorld
				rp->common.count++;
#else
				(((unsigned char *) rp)[0])++;
#endif
				if ((ap - b) > MaxReplyPacket) {
				    break;
				}
			    }
			    lr = lr->next;
			    scandex++;
			}

			l = ap - b;

			rp->common.length = htons(((l + 3) / 4) - 1);
			l = (ntohs(rp->common.length) + 1) * 4;
#ifdef HEXDUMP
			if (hexdump) {
                            fprintf(stderr, "%s: %d bytes sent to socket.\n", prog, l);
			    xd(stderr, b, l, TRUE);
			}
#endif
			if (send(csock, b, l, 0) < 0) {
                            perror("sending query match reply");
			}
		    }

                    /*  SFms  --  Retrieve the server's information
                                  message, if any.  If the server doesn't
				  publish a message, a null string is
				  returned in the reply packet.  */

                    else if (bcmp(p + 8, "SFms", 4) == 0) {
#ifdef HEXDUMP
			if (hexdump) {
                            fprintf(stderr, "%s: %d bytes sent to socket.\n",
				    prog, messagel);
			    xd(stderr, message, messagel, TRUE);
			}
#endif
			if (prolix) {
                            printf("%s: %s server message request\n", prog, inet_ntoa(from.sin_addr));
			}
			if (send(csock, message, messagel, 0) < 0) {
                            perror("sending server message");
			}
		    }
		    break;

		default:
		    if (debugging || verbose) {
                        fprintf(stderr, "Bogus payload type %d\n",
#ifdef RationalWorld
				      rp->common.pt
#else
				      ((unsigned char *) rp)[1]
#endif
			       );
		    }
		    break;
	    }
	}

	close(csock);
    }
#ifdef MEANS_OF_EXIT
    close(sock);
    return 0;
#endif
}
