/* idea.c - C source code for IDEA block cipher. IDEA (International Data 
 * Encryption Algorithm), formerly known as IPES (Improved Proposed Encryption
 * Standard). Algorithm developed by Xuejia Lai and James L. Massey, of ETH 
 * Zurich. This implementation modified and derived from original C code 
 * developed by Xuejia Lai. Zero-based indexing added, names changed from IPES
 * to IDEA. CFB functions added. Random number routines added. Optimized for 
 * speed 21 Oct 92 by Colin Plumb <colin@nsq.gts.org>. This code assumes that 
 * each pair of 8-bit bytes comprising a 16-bit word in the key and in the 
 * cipher block are externally represented with the Most Significant Byte 
 * (MSB) first, regardless of internal native byte order of the target CPU.  */

#ifdef TEST
#include <stdio.h>
#include <time.h>
#endif

#ifdef sgi
#define HIGHFIRST
#endif

#ifdef sun
#define HIGHFIRST
#define const
#endif

#include "idea.h"

#define FAR	IFAR

#ifndef _WIN32
#ifdef _M_I86
#define USE_ASM
#endif
#else
#define IDEA32
#endif

#define min(x, y) (((x) < (y)) ? (x) : (y))

#define TRUE	1
#define FALSE	0

#define IDEABLOCKSIZE 8
#define ROUNDS      8               /* Don't change this value, should be 8 */
#define KEYLEN		(6*ROUNDS+4)	/* length of key schedule */

#define byte	unsigned char
#define word16	unsigned short
#define boolean int
#define word32	unsigned long
#define byteptr unsigned char FAR *

typedef word16 IDEAkey[KEYLEN];

#ifdef IDEA32 /* Use >16-bit temporaries */
#define low16(x) ((x) & 0xFFFF)
typedef unsigned int uint16;		/* at LEAST 16 bits, maybe more */
#else
#define low16(x) (x)                /* this is only ever applied to uint16's */
typedef word16 uint16;
#endif

#ifdef _GNUC_
/* __const__ simply means there are no side effects for this function,
 * which is useful info for the gcc optimizer */
#define CONST __const__
#else
#define CONST
#endif

static void en_key_idea(word16 *userkey, word16 *Z);
static void de_key_idea(IDEAkey Z, IDEAkey DK);

/* Multiplication, modulo (2**16)+1. Note that this code is structured like 
 * this on the assumption that untaken branches are cheaper than taken 
 * branches, and the compiler doesn't schedule branches. */

#ifdef SMALL_CACHE
CONST static uint16 mul(register uint16 a, register uint16 b)
{
	   register word32 p;
	   if (a)
	   {	 if (b)
			 {		p = (word32)a * b;
					b = low16(p);
					a = p>>16;
					return b - a + (b < a);
			 }
			 else
			 {		return 1-a;
			 }
	   }
	   else
	   {	 return 1-b;
	   }
}
#endif /* SMALL_CACHE */

/* Compute multiplicative inverse of x, modulo (2**16)+1, using Euclid's GCD 
 * algorithm. It is unrolled twice to avoid swapping the meaning of the 
 * registers each iteration; some subtracts of t have been changed to adds.  */

CONST static uint16 inv(uint16 x)	   
{
	   uint16 t0, t1;
	   uint16 q, y;
	   if (x <= 1)
			 return x;	  /* 0 and 1 are self-inverse */
	   t1 = 0x10001 / x;  /* Since x >= 2, this fits into 16 bits */
	   y = 0x10001 % x;
	   if (y == 1)
			 return low16(1-t1);
	   t0 = 1;
	   do
	   {	 q = x / y;
			 x = x % y;
			 t0 += q * t1;
			 if (x == 1)
					return t0;
			 q = y / x;
			 y = y % x;
			 t1 += q * t0;
	   } while (y != 1);
	   return low16(1-t1);
}

/*	   Compute IDEA encryption subkeys Z */

static void en_key_idea(word16 *userkey, word16 *Z)
{
	   int i,j;
	   /* shifts */
	   for (j=0; j<8; j++)
			 Z[j] = *userkey++;
	   for (i=0; j<KEYLEN; j++)
	   {	 i++;
			 Z[i+7] = Z[i & 7] << 9 | Z[i+1 & 7] >> 7;
			 Z += i & 8;
			 i &= 7;
	   }
}

/*	   Compute IDEA decryption subkeys DK from encryption subkeys Z */
/* Note: these buffers *may* overlap! */

static void de_key_idea(IDEAkey Z, IDEAkey DK)
{
	   int j;
	   uint16 t1, t2, t3;
	   IDEAkey T;
	   word16 *p = T + KEYLEN;
	   t1 = inv(*Z++);
	   t2 = -*Z++;
	   t3 = -*Z++;
	   *--p = inv(*Z++);
	   *--p = t3;
	   *--p = t2;
	   *--p = t1;
	   for (j = 1; j < ROUNDS; j++)
	   {
			 t1 = *Z++;
			 *--p = *Z++;
			 *--p = t1;
			 t1 = inv(*Z++);
			 t2 = -*Z++;
			 t3 = -*Z++;
			 *--p = inv(*Z++);
			 *--p = t2;
			 *--p = t3;
			 *--p = t1;
	   }
	   t1 = *Z++;
	   *--p = *Z++;
	   *--p = t1;
	   t1 = inv(*Z++);
	   t2 = -*Z++;
	   t3 = -*Z++;
	   *--p = inv(*Z++);
	   *--p = t3;
	   *--p = t2;
	   *--p = t1;
/* Copy and destroy temp copy */
	   for (j = 0, p = T; j < KEYLEN; j++)
	   {
			 *DK++ = *p;
			 *p++ = 0;
	   }
}

/* MUL(x,y) computes x = x*y, modulo 0x10001. Requires two temps, t16 and t32.
 * x must me a side-effect-free lvalue. y may be anything, but unlike x, must 
 * be strictly 16 bits even if low16() is #defined. All of these are 
 * equivalent; see which is faster on your machine.  */

#ifdef SMALL_CACHE
#define MUL(x,y) (x = mul(low16(x),y))
#else
#ifdef AVOID_JUMPS
#define MUL(x,y) (x = low16(x-1), t16 = low16((y)-1), \
             t32 = (word32)x*t16+x+t16+1, x = low16(t32), \
			 t16 = t32>>16, x = x-t16+(x<t16) )
#else
#define MUL(x,y) ((t16 = (y)) ? (x=low16(x)) ? \
        t32 = (word32)x*t16, x = low16(t32), t16 = t32>>16, \
        x = x-t16+(x<t16) : \
		(x = 1-t16) : (x = 1-x))
#endif
#endif

#ifdef USE_ASM

static void cipher_idea(word16 FAR *inblock, word16 FAR *outblock, IDEAkey zkey)
{
	word16 sx1, sx4, skk, done8;
	__asm {
;A while ago I posted a message claiming a speed of 238,000
;bytes/sec for an implementation of IDEA on a 33Mh 486.  Below is
;an explanation and some code to show how it works.  The basic
;trick should be useful on many (but not all) processors.  I
;expect only those familiar with IDEA and its reference
;implementation will be able to follow the discussion.	See:
;
;Lai, Xueja and Massey, James L.  A Proposal for a New Block
;Encryption Standard, Eurocrypt 90
;
;For those who have been asking for the code, sorry I kept
;putting it off.  I wanted to get it out of Turbo Pascal
;ideal-mode, but I never had the time.
;
;Colin Plum wrote IDEA-386 code which is included in PGP
;2.3a and uses the same tricks.  I don't know who's is
;faster, but I expect they will be very close.	Now
;here's how it's done.
;
;A major bottleneck in software IDEA is the mul() routine, which
;is used 34 times per 64 bit block.  The routine performs
;multiplication in the multiplicative group mod 2^16+1.  The two
;factors are each in a 16 bit word, and the output is also in a 16
;bit word.	Note that 0 is not a member of the multiplicative
;group and 2^16 does not fit in 16 bits. We therefor use the 0
;word to represent 2^16.  Now group elements map one to one onto
;all possible 16 bit words, since 2^16+1 is prime.
;
;Here is (essentially) the reference implementation from [Lai].
;
;
;unsigned mul( unsigned a, unsigned b ) {
;  long int p ;
;  long unsigned q ;
;		if( a==0 ) p= 0x00010001 - b ;
;		else if( b==0 ) p= 0x00010001 - a ;
;		else {
;				q= a*b;
;				p= (q & 0xffff) - (q>>16)
;				if( p<0 ) p= p + 0x00010001 ;
;		  }
;		return (unsigned)(p & 0xffff) ;
;}
;
;
;Note the method of reducing a 32 bit word modulo 2^16-1.  We
;subtract the high word from the low word, and add the modulus
;back if the result is less than 0.  [Lai] contains a proof that
;this works, and you can convince yourself fairly easily.
;
;To speed up this routine, we note that the tests for a=0 and b=0
;will rarely be false.	With the possible exception of the first 2
;of the 34 multiplications, 0 should be no more likely than any of
;the other 65535 numbers.  Note that if (and only if) either a or
;b is 0 then q will also be 0, and we can check for this in one
;instruction if our processor sets a zero flag for multiplication
;(as the 68000 does but 80x86 does not).
;
;Fortunately p will also be zero after the subtraction if and only
;if either a or b is 0.  Proof: r will be zero when the high order
;word of q equals the low order word, and that happens when q is
;divisible by 00010001 hex.  Since 00010001h = 2^16+1 is prime,
;this happens if either a or b is a multiple of 2^16+1, and 0 is
;the only such multiple which will fit in a 16 bit word.
;
;The speed-up strategy is to proceed under the assumption that a
;and b are not 0, check to be sure in one instruction, and
;recompute if the assumption was wrong.  Here's some 8086
;assembler code:
;
;		mov  ax, [a]
;		mul  [b]		; ax is implied. q is now in DX AX
;		sub  ax, dx 	; mod 2^16+1
;		jnz  not0		; Jump if neither op was 0. Usually taken.
;
;		mov  ax, 1		; recompute result knowing one op is 0.
;		sub  ax, [a]
;		sub  ax, [b]
;		jmp  out		; Just jump over adding the carry.
;not0:
;		adc  ax, 0		; If r<0 add 1, otherwise do nothing.
;out:				 ; Result is now in ax
;
;
;Note that when r<0 we add 1 instead of 2^16+1 since the 2^16 part
;overflows out of the result.  The "adc  ax, 0" does all the work
;of checking for a negative result and adding the modulus if
;needed.
;
;The multiplication takes 9 instructions, 4 of which are rarely
;executed.	I believe similar tricks are possible on many
;processors.  The one drawback to the check-after-multiply tactic
;is that we can't let the multiply overwrite the only copy of an
;operand.
;
;Note that most software implementations of IDEA will run at
;slightly different speeds when 0's come up in the multiply
;routine.  The reference implementation is faster on 0, this one
;is faster on non-zero.  This may be a problem for some real-time
;stuff, and also suggests an attack based on timing.
;
;Finally, below is an implementation of the complete encryption
;function in 8086 assembler, to replace the cipher_idea() function
;in PGP.  It takes the same parameters as the function from PGP,
;and uses the c language calling conventions.  I tested it using
;the debug features of the idea.c file in PGP.	You will need to
;add segment/assume directives.  This version uses no global data
;and should be reentrant.
;
;The handling of zero multipliers is outside the inner loop so
;that a short conditional jump can loop back to the beginning.
;Forward conditional jumps are usually not taken and backward
;jumps are usually taken, which is consistent with 586 branch
;prediction (or so I've heard).  Stalls where the output of one
;instruction is needed for the next seem unavoidable.
;
;Last I heard, IDEA was patent pending.  My code is up for grabs,
;although I would get a kick out being credited if you use it.
;On the other hand Colin's code is already tested and ready
;to assemble and link with PGP.
;
;--Bryan
;
;____________________CODE STARTS BELOW THIS LINE_________

;  Called as: cipher_idea(inbuff, outbuff, zkey)
;  All arguments must be near pointers addressed off DS.

 ;		push ax 	; My compiler assumes these are not saved.
 ;		push bx
 ;		push cx
 ;		push dx

		push si
		push di

; Put the 16 bit sub-blocks in registers and/or local variables
		mov  si, [inblock]
		mov  ax, [si]
		mov  [sx1], ax		 ; x1  is in ax and sx1
		mov  di, [si+2] 	 ; x2  is in di
		mov  bx, [si+4] 	 ; x3  is in bx
		mov  dx, [si+6]
		mov  [sx4], dx		 ; x4  is in sx4

		mov  si, [zkey] 	 ; si points to next subkey
		mov  [done8], si
		add  [done8], 96	 ; we will be finished with 8 rounds
							 ; when si=done8

LLloop: 					 ; 8 rounds of this
		add  di, [si+2] 	 ; x2+=zkey[2]	is in di
		add  bx, [si+4] 	 ; x3+=zkey[4]	is in bx

		mul  [Word  ptr si]	 ;x1 *= zkey[0]
		sub  ax, dx
		jz	LLx1			 ; if 0, use special case multiply
		adc  ax, 0
LLx1out:
		mov  [sx1], ax		 ; x1 is in ax and sx1

		xor  ax, bx 		 ; ax= x1^x3
		mul  [Word ptr si+8] ; compute kk
		sub  ax, dx 		 ; if 0, use special case multiply
		jz	LLkk
		adc  ax, 0
LLkkout:
		mov  cx, ax 		 ; kk is in cx

		mov  ax, [sx4]		 ; x4 *= zkey[6]
		mul  [Word ptr si+6]
		sub  ax, dx
		jz	 LLx4			 ; if 0, use special case multiply
		adc  ax, 0
LLx4out:
		mov  [sx4], ax		 ; x4 is in sx4 and ax

		xor  ax, di 		 ; x4^x2
		add  ax, cx 		 ; kk+(x2^x4)
		mul  [Word ptr si+10]; compute t1
		sub  ax, dx
		jz	LLt1			 ; if 0, use special case multiply
		adc  ax, 0
LLt1out:					 ; t1 is in ax

		add  cx, ax 		 ; t2 is in cx	 kk+t1

		xor  [sx4], cx		 ; x4 in sx4
		xor  di, cx 		 ; new x3 in di
		xor  bx, ax 		 ; new x2 in bx
		xchg bx, di 		 ; x2 in di, x3 in bx
		xor  ax, [sx1]		 ; x1 in ax
		mov  [sx1], ax		 ; and [sx1]

		add  si, 12 		 ; point to next subkey
		cmp  si, [done8]
		jne  LLloop
		jmp  LLout8

;------------------------------------------
; Special case multiplications, when one factor is 0

LLx1:	mov  ax, 1
		sub  ax, [sx1]
		sub  ax, [Word ptr si]
		jmp  LLx1out

LLkk:	mov  ax, [sx1]		 ; rebuild overwritten operand
		xor  ax, bx
		neg  ax
		inc  ax
		sub  ax, [si+8]
		jmp  LLkkout

LLx4:	mov  ax, 1
		sub  ax, [sx4]
		sub  ax, [Word ptr si+6]
		jmp  LLx4out

LLt1:	mov  ax, [sx4]		 ; rebuild
		xor  ax, di
		add  ax, cx
		neg  ax
		inc  ax
		sub  ax, [si+10]
		jmp  LLt1out

;---------------------------------------------------
;	8 rounds are done, now that extra pseudo-round

LLout8:
		push di
		mov  di, [outblock]

		mul  [Word ptr si]
		sub  ax, dx
		jnz  LLo1n			 ; jump over special case code
		mov  ax, 1
		sub  ax, [sx1]
		sub  ax, [si]
		jmp  LLo1out
LLo1n:	adc  ax, 0
LLo1out:  mov [di], ax		 ; final ciphertext block 1

		mov  ax, [sx4]
		mul  [Word ptr si+6]
		sub  ax, dx
		jnz  LLo4n			 ; jump over special case code
		mov  ax, 1
		sub  ax, [sx4]
		sub  ax, [si+6]
		jmp  LLo4out
LLo4n:	adc  ax, 0
LLo4out: mov  [di+6], ax	 ; final ciphertext block 4

		add  bx, [si+2]
		mov  [di+2], bx 	 ; final ciphertext block 2
		pop  ax
		add  ax, [si+4]
		mov  [di+4], ax 	 ; final ciphertext block 3

;  Restore the stack and return

		pop  di
		pop  si
;		pop  dx
;		pop  cx
;		pop  bx
;		pop  ax
	}
}

#else

/* IDEA encryption/decryption algorithm. In/out can be the same buffer */ 

static void cipher_idea(word16 FAR *in, word16 FAR *out, register CONST IDEAkey Z)
{
	   register uint16 x1, x2, x3, x4, t1, t2;
	   register uint16 t16;
	   register word32 t32;
	   int r = ROUNDS;
	   x1 = *in++;	x2 = *in++;
	   x3 = *in++;	x4 = *in;
	   do
	   {
			 MUL(x1,*Z++);
			 x2 += *Z++;
			 x3 += *Z++;
			 MUL(x4, *Z++);
			 t2 = x1^x3;
			 MUL(t2, *Z++);
			 t1 = t2 + (x2^x4);
			 MUL(t1, *Z++);
			 t2 = t1+t2;
			 x1 ^= t1;
			 x4 ^= t2; 
			 t2 ^= x2;
			 x2 = x3^t1;
			 x3 = t2;
	   } while (--r);
	   MUL(x1, *Z++);
	   *out++ = x1;
	   *out++ = x3 + *Z++;
	   *out++ = x2 + *Z++;
	   MUL(x4, *Z);
	   *out = x4;
}
#endif

#ifdef TEST

/* Number of Kbytes of test data to encrypt. Defaults to 1 MByte. */

#ifndef KBYTES
#define KBYTES 1024
#endif

#ifndef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC 1000000
#endif

int main()
{	   /* Test driver for IDEA cipher */ 
	   int i, j, k; 
	   IDEAkey Z, DK;
	   word16 XX[4], TT[4], YY[4];	   
	   word16 userkey[8];
	   clock_t start, end;
	   long l;

	   /* Make a sample user key for testing... */

	   for(i=0; i<8; i++)
			 userkey[i] = i+1;

	   /* Compute encryption subkeys from user key... */

	   en_key_idea(userkey,Z);
       printf("\nEncryption key subblocks: ");
	   for(j=0; j<ROUNDS+1; j++)
	   {
             printf("\nround %d:   ", j+1);
			 if (j==ROUNDS)
					for(i=0; i<4; i++)
                          printf(" %6u", Z[j*6+i]);
			 else
					for(i=0; i<6; i++)
                          printf(" %6u", Z[j*6+i]);
	   }

	   /* Compute decryption subkeys from encryption subkeys... */

	   de_key_idea(Z,DK);
       printf("\nDecryption key subblocks: ");
	   for(j=0; j<ROUNDS+1; j++)
	   {
             printf("\nround %d:   ", j+1);
			 if (j==ROUNDS)
					for(i=0; i<4; i++)
                          printf(" %6u", DK[j*6+i]);
			 else
					for(i=0; i<6; i++)
                          printf(" %6u", DK[j*6+i]);
	   }

	   /* Make a sample plaintext pattern for testing... */

	   for (k=0; k<4; k++)
			 XX[k] = k;
       printf("\n Encrypting %d KBytes (%ld blocks)...", KBYTES, KBYTES*64l);
	   fflush(stdout);
	   start = clock();
	   cipher_idea(XX,YY,Z);		 /* encrypt plaintext XX, making YY */ 
	   for (l = 1; l < 64*KBYTES; l++)
			 cipher_idea(YY,YY,Z);	 /* repeated encryption */
	   cipher_idea(YY,TT,DK);		 /* decrypt ciphertext YY, making TT */ 
	   for (l = 1; l < 64*KBYTES; l++)
			 cipher_idea(TT,TT,DK);  /* repeated decryption */
	   end = clock() - start;
	   l = end * 1000. / CLOCKS_PER_SEC + 1;
	   i = l/1000;
	   j = l%1000;
	   l = KBYTES * 1024. * CLOCKS_PER_SEC / end;

       printf("%d.%03d seconds = %ld bytes per second\n", i, j, l);
       printf("\nX %6u   %6u  %6u  %6u \n",    
		 XX[0], XX[1],	XX[2], XX[3]);
       printf("Y %6u   %6u  %6u  %6u \n",
		 YY[0], YY[1],	YY[2], YY[3]);
       printf("T %6u   %6u  %6u  %6u \n",
		 TT[0], TT[1],	TT[2], TT[3]);
	   /* Now decrypted TT should be same as original XX */
	   for (k=0; k<4; k++)
			 if (TT[k] != XX[k])
			 {
                    printf("\n\07Error!  Noninvertable encryption.\n");
					exit(-1);	   /* error exit */ 
			 }
       printf("\nNormal exit.\n");
	   return 0;
}
#endif /* TEST */

/* xorbuf - change buffer via xor with random mask block. Used for Cipher 
 * Feedback (CFB) or Cipher Block Chaining (CBC) modes of encryption. Can be
 *	applied for any block encryption algorithm, with any block size, such as 
 * the DES or the IDEA cipher.	*/

static void xorbuf(register byteptr buf, register byteptr mask, register int count)
/*	   count must be > 0 */
{
	   if (count) 
			 do
					*buf++ ^= *mask++;
			 while (--count);
}	   /* xorbuf */

/* cfbshift - shift bytes into IV for CFB input. Used only for Cipher Feedback
 * (CFB) mode of encryption. Can be applied for any block encryption algorithm
 * with any block size, such as the DES or the IDEA cipher.  */

static void cfbshift(register byteptr iv, register byteptr buf, register int count, int blocksize)
/* iv is initialization vector. buf is buffer pointer. count is number of bytes
 * to shift in...must be > 0. blocksize is 8 bytes for DES or IDEA ciphers. */
{
	   int retained;
	   if (count)
	   {
			 retained = blocksize-count;   /* number bytes in iv to retain */
		 /* left-shift retained bytes of IV over by count bytes to make room */
			 while (retained--)
			 {
					*iv = *(iv+count);
					iv++;
			 }
			 /* now copy count bytes from buf to shifted tail of IV */
			 do 	*iv++ = *buf++;
			 while (--count);
	   }
}	   /* cfbshift */

/* Key schedules for IDEA encryption and decryption */

static IDEAkey Z, DK;
static word16 FAR *iv_idea; 	/* pointer to IV for CFB or CBC */
static boolean cfb_dc_idea;		/* TRUE iff CFB decrypting */

/* initkey_idea initializes IDEA for ECB mode operations */

void initkey_idea(byte FAR *key, boolean decryp)
{
	   word16 userkey[8]; /* IDEA key is 16 bytes long */
	   int i;
	   /* Assume each pair of bytes comprising a word is ordered MSB-first. */
	   for (i=0; i<8; i++)
	   {
			 userkey[i] = (key[0]<<8) + key[1];
			 key++; key++;
	   }
	   en_key_idea(userkey,Z);
	   if (decryp)
	   {
			 de_key_idea(Z,Z);	 /* compute inverse key schedule DK */
	   }
	   for (i=0; i<8; i++)/* Erase dangerous traces */
			 userkey[i] = 0;
} /* initkey_idea */

/* Run a 64-bit block thru IDEA in ECB (Electronic Code Book) mode, using the
 * currently selected key schedule. */

void idea_ecb(word16 FAR *inbuf, word16 FAR *outbuf)
{
	   /* Assume each pair of bytes comprising a word is ordered MSB-first. */
#ifndef HIGHFIRST	/* If this is a least-significant-byte-first CPU */
	   word16 x;
	   /* Invert the byte order for each 16-bit word for internal use. */
	   x = inbuf[0]; outbuf[0] = x >> 8 | x << 8;
	   x = inbuf[1]; outbuf[1] = x >> 8 | x << 8;
	   x = inbuf[2]; outbuf[2] = x >> 8 | x << 8;
	   x = inbuf[3]; outbuf[3] = x >> 8 | x << 8;
	   cipher_idea(outbuf, outbuf, Z);
	   x = outbuf[0]; outbuf[0] = x >> 8 | x << 8;
	   x = outbuf[1]; outbuf[1] = x >> 8 | x << 8;
	   x = outbuf[2]; outbuf[2] = x >> 8 | x << 8;
	   x = outbuf[3]; outbuf[3] = x >> 8 | x << 8;
#else  /* HIGHFIRST */
	   /* Byte order for internal and external representations is the same. */
	   cipher_idea(inbuf, outbuf, Z);
#endif /* HIGHFIRST */
} /* idea_ecb */

/* initcfb - Initializes IDEA key schedule tables via key; initializes Cipher
 * Feedback mode IV. References context variables cfb_dc_idea and iv_idea.	*/

void initcfb_idea(word16 FAR *iv0, byte FAR *key, boolean decryp)
/* iv0 is copied to global iv_idea, buffer will be destroyed by ideacfb. key is
 * pointer to key buffer. decryp is TRUE if decrypting, FALSE if encrypting. */
{
	   iv_idea = iv0;
	   cfb_dc_idea = decryp;
	   initkey_idea(key,FALSE);
}

/* ideacfb - encipher a buffer with IDEA enciphering algorithm, using Cipher
 *	Feedback (CFB) mode. Assumes initcfb_idea has already been called. 
 * References context variables cfb_dc_idea and iv_idea.  */

void ideacfb(byteptr buf, int count)
/* buf is input, output buffer, may be more than 1 block. count is byte count
 * is byte count of buffer.  May be > IDEABLOCKSIZE. */
{
	   int chunksize;				 /* smaller of count, IDEABLOCKSIZE */
	   word16 temp[IDEABLOCKSIZE/2];

	   while ((chunksize = min(count,IDEABLOCKSIZE)) > 0)
	   {
			 idea_ecb(iv_idea,temp);  /* encrypt iv_idea, making temp. */ 
			 if (cfb_dc_idea)/* buf is ciphertext */
					/* shift in ciphertext to IV... */
					cfbshift((byte FAR *)iv_idea,buf,chunksize,IDEABLOCKSIZE);
			 /* convert buf via xor */
			 xorbuf(buf,(byte *)temp,chunksize);/* buf has enciphered output */
			 if (!cfb_dc_idea)/* buf was plaintext, is now ciphertext */
					/* shift in ciphertext to IV... */
					cfbshift((byte FAR *)iv_idea,buf,chunksize,IDEABLOCKSIZE);
			 count -= chunksize;
			 buf += chunksize;
	   }
}

/* close_idea function erases all the key schedule information when we are 
 * done with a set of operations for a particular IDEA key context. This is to
 * prevent any sensitive data from being left around in memory. */

void close_idea(void)		/* erase current key schedule tables */
{
	   short i;
	   for (i = 0; i < KEYLEN; i++)
			 Z[i] = 0;
}	   /* close_idea() */

/* These buffers are used by init_idearand, idearand, and close_idearand. */
static word16 dtbuf_idea[4] = {0};	   /* buffer for enciphered timestamp */
static word16 randseed_idea[4] = {0};  /* seed for IDEA random # generator */
static word16 randbuf_idea[4] = {0};   /* buffer for IDEA random # generator */
static byte randbuf_idea_counter = 0;  /* random bytes left in randbuf_idea */

/* init_idearand - initialize idearand, IDEA random number generator. Used for
 *	generating cryptographically strong random numbers. Much of design comes 
 * from Appendix C of ANSI X9.17. key is pointer to IDEA key buffer. seed is 
 * pointer to random number seed buffer. tstamp is a 32-bit timestamp */

void init_idearand(byte key[16], byte seed[8], word32 tstamp)
{
	   int i;
	   initkey_idea(key, FALSE);	  /* initialize IDEA */
	   for (i=0; i<4; i++)			  /* capture timestamp material */
	   {	 dtbuf_idea[i] = (word16) tstamp;  /* get bottom word */
			 tstamp = tstamp >> 16;   /* drop bottom word */
			 /* tstamp has only 4 bytes-- last 4 bytes will always be 0 */
	   }
	   /* Start with enciphered timestamp: */
	   idea_ecb(dtbuf_idea,dtbuf_idea);
	   /* initialize seed material */
	   for (i=0; i<8; i++)
			 ((byte *)randseed_idea)[i] = seed[i];
	   randbuf_idea_counter = 0; /* # of random bytes left in randbuf_idea */
}

/* idearand - IDEA pseudo-random number generator. Used for generating 
 * cryptographically strong random numbers. Much of design comes from Appendix
 * C of ANSI X9.17.  */

byte idearand(void)
{
	   int i;
	   if (randbuf_idea_counter==0) 		/* if random buffer is spent...*/
	   {	 /* Combine enciphered timestamp with seed material: */
			 for (i=0; i<4; i++)
					randseed_idea[i] ^= dtbuf_idea[i];
			 idea_ecb(randseed_idea,randbuf_idea);	 /* fill new block */
			 /* Compute new seed vector: */
			 for (i=0; i<4; i++)
					randseed_idea[i] = randbuf_idea[i] ^ dtbuf_idea[i];
			 idea_ecb(randseed_idea,randseed_idea);    /* fill new seed */
			 randbuf_idea_counter = 8;	 /* reset counter for full buffer */
	   }
	   /* Take a byte from randbuf_idea: */
	   return(((byte *)randbuf_idea)[--randbuf_idea_counter]);
}

void close_idearand(void)
{	   /* Erase random IDEA buffers and wipe out IDEA key info */
	   int i;
	   for (i=0; i<4; i++)
	   {	 randbuf_idea[i] = 0;
			 randseed_idea[i] = 0;
			 dtbuf_idea[i] = 0;
	   }
	   close_idea();/* erase current key schedule tables */
}
